/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree.CoreNode;
import org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree.HTConfig;
import org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree.HTInterval;
import org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree.HTNode;
import org.eclipse.linuxtools.internal.tmf.core.statesystem.backends.historytree.HT_IO;
import org.eclipse.linuxtools.tmf.core.exceptions.TimeRangeException;

class HistoryTree {
    public static final int TREE_HEADER_SIZE = 4096;
    private static final int HISTORY_FILE_MAGIC_NUMBER = 100641024;
    private static final int FILE_VERSION = 3;
    private final HTConfig config;
    private final HT_IO treeIO;
    private long treeEnd;
    private int nodeCount;
    private List<CoreNode> latestBranch;
    private int curDepth;

    HistoryTree(HTConfig conf) throws IOException {
        if (conf.getBlockSize() < 4096) {
            throw new IllegalArgumentException();
        }
        this.config = conf;
        this.treeEnd = conf.getTreeStart();
        this.nodeCount = 0;
        this.latestBranch = new ArrayList<CoreNode>();
        this.treeIO = new HT_IO(this, true);
        CoreNode firstNode = this.initNewCoreNode(-1, conf.getTreeStart());
        this.latestBranch.add(firstNode);
    }

    HistoryTree(File existingStateFile, int expProviderVersion) throws IOException {
        if (!existingStateFile.exists()) {
            throw new IOException("Selected state file does not exist");
        }
        if (existingStateFile.length() <= 0L) {
            throw new IOException("Empty target file");
        }
        FileInputStream fis = new FileInputStream(existingStateFile);
        ByteBuffer buffer = ByteBuffer.allocate(4096);
        FileChannel fc = fis.getChannel();
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        fc.read(buffer);
        buffer.flip();
        int res = buffer.getInt();
        if (res != 100641024) {
            fc.close();
            fis.close();
            throw new IOException("Wrong magic number");
        }
        res = buffer.getInt();
        if (res != 3) {
            fc.close();
            fis.close();
            throw new IOException("Mismatching History Tree file format versions");
        }
        res = buffer.getInt();
        if (res != expProviderVersion && expProviderVersion != -42) {
            fc.close();
            fis.close();
            throw new IOException("Mismatching event handler versions");
        }
        int bs = buffer.getInt();
        int maxc = buffer.getInt();
        this.nodeCount = buffer.getInt();
        int rootNodeSeqNb = buffer.getInt();
        long startTime = buffer.getLong();
        this.config = new HTConfig(existingStateFile, bs, maxc, expProviderVersion, startTime);
        fc.close();
        fis.close();
        this.treeIO = new HT_IO(this, false);
        this.rebuildLatestBranch(rootNodeSeqNb);
        this.treeEnd = this.latestBranch.get(0).getNodeEnd();
        if (startTime != this.latestBranch.get(0).getNodeStart()) {
            fc.close();
            fis.close();
            throw new IOException("Inconsistent start times in thehistory file, it might be corrupted.");
        }
    }

    void closeTree(long requestedEndTime) {
        this.treeEnd = requestedEndTime;
        int i = 0;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(this.treeEnd);
            this.treeIO.writeNode(this.latestBranch.get(i));
            ++i;
        }
        FileChannel fc = this.treeIO.getFcOut();
        ByteBuffer buffer = ByteBuffer.allocate(4096);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        try {
            try {
                fc.position(0L);
                buffer.putInt(100641024);
                buffer.putInt(3);
                buffer.putInt(this.config.getProviderVersion());
                buffer.putInt(this.config.getBlockSize());
                buffer.putInt(this.config.getMaxChildren());
                buffer.putInt(this.nodeCount);
                buffer.putInt(this.latestBranch.get(0).getSequenceNumber());
                buffer.putLong(this.latestBranch.get(0).getNodeStart());
                buffer.flip();
                int res = fc.write(buffer);
                assert (res <= 4096);
            }
            catch (IOException iOException) {
                try {
                    fc.close();
                }
                catch (IOException iOException2) {}
            }
        }
        finally {
            try {
                fc.close();
            }
            catch (IOException iOException) {}
        }
    }

    HTConfig getConfig() {
        return this.config;
    }

    long getTreeStart() {
        return this.config.getTreeStart();
    }

    long getTreeEnd() {
        return this.treeEnd;
    }

    int getNodeCount() {
        return this.nodeCount;
    }

    HT_IO getTreeIO() {
        return this.treeIO;
    }

    List<CoreNode> getLatestBranch() {
        return Collections.unmodifiableList(this.latestBranch);
    }

    private void rebuildLatestBranch(int rootNodeSeqNb) throws ClosedChannelException {
        this.latestBranch = new ArrayList<CoreNode>();
        HTNode nextChildNode = this.treeIO.readNodeFromDisk(rootNodeSeqNb);
        this.latestBranch.add((CoreNode)nextChildNode);
        while (this.latestBranch.get(this.latestBranch.size() - 1).getNbChildren() > 0) {
            nextChildNode = this.treeIO.readNodeFromDisk(this.latestBranch.get(this.latestBranch.size() - 1).getLatestChild());
            this.latestBranch.add((CoreNode)nextChildNode);
        }
    }

    void insertInterval(HTInterval interval) throws TimeRangeException {
        if (interval.getStartTime() < this.config.getTreeStart()) {
            throw new TimeRangeException();
        }
        this.tryInsertAtNode(interval, this.latestBranch.size() - 1);
    }

    private void tryInsertAtNode(HTInterval interval, int indexOfNode) {
        HTNode targetNode = this.latestBranch.get(indexOfNode);
        if (interval.getIntervalSize() > targetNode.getNodeFreeSpace()) {
            this.addSiblingNode(indexOfNode);
            this.tryInsertAtNode(interval, this.latestBranch.size() - 1);
            return;
        }
        if (interval.getStartTime() < targetNode.getNodeStart()) {
            assert (indexOfNode >= 1);
            this.tryInsertAtNode(interval, indexOfNode - 1);
            return;
        }
        targetNode.addInterval(interval);
        if (interval.getEndTime() > this.treeEnd) {
            this.treeEnd = interval.getEndTime();
        }
    }

    private void addSiblingNode(int indexOfNode) {
        long splitTime = this.treeEnd;
        assert (indexOfNode < this.latestBranch.size());
        if (indexOfNode == 0) {
            this.addNewRootNode();
            return;
        }
        if (this.latestBranch.get(indexOfNode - 1).getNbChildren() == this.config.getMaxChildren()) {
            this.addSiblingNode(indexOfNode - 1);
            return;
        }
        int i = indexOfNode;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(splitTime);
            this.treeIO.writeNode(this.latestBranch.get(i));
            CoreNode prevNode = this.latestBranch.get(i - 1);
            CoreNode newNode = this.initNewCoreNode(prevNode.getSequenceNumber(), splitTime + 1L);
            prevNode.linkNewChild(newNode);
            this.latestBranch.set(i, newNode);
            ++i;
        }
    }

    private void addNewRootNode() {
        long splitTime = this.treeEnd;
        CoreNode oldRootNode = this.latestBranch.get(0);
        CoreNode newRootNode = this.initNewCoreNode(-1, this.config.getTreeStart());
        oldRootNode.setParentSequenceNumber(newRootNode.getSequenceNumber());
        int i = 0;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(splitTime);
            this.treeIO.writeNode(this.latestBranch.get(i));
            ++i;
        }
        newRootNode.linkNewChild(oldRootNode);
        int depth = this.latestBranch.size();
        this.latestBranch = new ArrayList<CoreNode>();
        this.latestBranch.add(newRootNode);
        i = 1;
        while (i < depth + 1) {
            CoreNode prevNode = this.latestBranch.get(i - 1);
            CoreNode newNode = this.initNewCoreNode(prevNode.getParentSequenceNumber(), splitTime + 1L);
            prevNode.linkNewChild(newNode);
            this.latestBranch.add(newNode);
            ++i;
        }
    }

    private CoreNode initNewCoreNode(int parentSeqNumber, long startTime) {
        CoreNode newNode = new CoreNode(this, this.nodeCount, parentSeqNumber, startTime);
        ++this.nodeCount;
        if (startTime >= this.treeEnd) {
            this.treeEnd = startTime + 1L;
        }
        return newNode;
    }

    HTNode selectNextChild(CoreNode currentNode, long t) throws ClosedChannelException {
        assert (currentNode.getNbChildren() > 0);
        int potentialNextSeqNb = currentNode.getSequenceNumber();
        int i = 0;
        while (i < currentNode.getNbChildren()) {
            if (t < currentNode.getChildStart(i)) break;
            potentialNextSeqNb = currentNode.getChild(i);
            ++i;
        }
        assert (potentialNextSeqNb != currentNode.getSequenceNumber());
        if (currentNode.isDone()) {
            return this.treeIO.readNodeFromDisk(potentialNextSeqNb);
        }
        return this.treeIO.readNode(potentialNextSeqNb);
    }

    long getFileSize() {
        return this.config.getStateFile().length();
    }

    boolean checkNodeIntegrity(HTNode zenode) {
        StringBuffer buf = new StringBuffer();
        boolean ret = true;
        if (!(zenode instanceof CoreNode)) {
            return true;
        }
        CoreNode node = (CoreNode)zenode;
        try {
            HTNode otherNode;
            if (node.getNbChildren() > 0) {
                otherNode = this.treeIO.readNode(node.getChild(0));
                if (node.getNodeStart() != otherNode.getNodeStart()) {
                    buf.append("Start time of node (" + node.getNodeStart() + ") " + "does not match start time of first child " + "(" + otherNode.getNodeStart() + "), " + "node #" + otherNode.getSequenceNumber() + ")\n");
                    ret = false;
                }
                if (node.isDone()) {
                    otherNode = this.treeIO.readNode(node.getLatestChild());
                    if (node.getNodeEnd() != otherNode.getNodeEnd()) {
                        buf.append("End time of node (" + node.getNodeEnd() + ") does not match end time of last child (" + otherNode.getNodeEnd() + ", node #" + otherNode.getSequenceNumber() + ")\n");
                        ret = false;
                    }
                }
            }
            int i = 0;
            while (i < node.getNbChildren()) {
                otherNode = this.treeIO.readNode(node.getChild(i));
                if (otherNode.getNodeStart() != node.getChildStart(i)) {
                    buf.append("  Expected start time of child node #" + node.getChild(i) + ": " + node.getChildStart(i) + "\n" + "  Actual start time of node #" + otherNode.getSequenceNumber() + ": " + otherNode.getNodeStart() + "\n");
                    ret = false;
                }
                ++i;
            }
        }
        catch (ClosedChannelException e) {
            e.printStackTrace();
        }
        if (!ret) {
            System.out.println("");
            System.out.println("SHT: Integrity check failed for node #" + node.getSequenceNumber() + ":");
            System.out.println(buf.toString());
        }
        return ret;
    }

    void checkIntegrity() {
        try {
            int i = 0;
            while (i < this.nodeCount) {
                this.checkNodeIntegrity(this.treeIO.readNode(i));
                ++i;
            }
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    public String toString() {
        return "Information on the current tree:\n\nBlocksize: " + this.config.getBlockSize() + "\n" + "Max nb. of children per node: " + this.config.getMaxChildren() + "\n" + "Number of nodes: " + this.nodeCount + "\n" + "Depth of the tree: " + this.latestBranch.size() + "\n" + "Size of the treefile: " + this.getFileSize() + "\n" + "Root node has sequence number: " + this.latestBranch.get(0).getSequenceNumber() + "\n" + "'Latest leaf' has sequence number: " + this.latestBranch.get(this.latestBranch.size() - 1).getSequenceNumber();
    }

    private void preOrderPrint(PrintWriter writer, boolean printIntervals, CoreNode currentNode) {
        writer.println(currentNode.toString());
        if (printIntervals) {
            currentNode.debugPrintIntervals(writer);
        }
        ++this.curDepth;
        try {
            int i = 0;
            while (i < currentNode.getNbChildren()) {
                HTNode nextNode = this.treeIO.readNode(currentNode.getChild(i));
                assert (nextNode instanceof CoreNode);
                int j = 0;
                while (j < this.curDepth - 1) {
                    writer.print("  ");
                    ++j;
                }
                writer.print("+-");
                this.preOrderPrint(writer, printIntervals, (CoreNode)nextNode);
                ++i;
            }
        }
        catch (ClosedChannelException e) {
            e.printStackTrace();
        }
        --this.curDepth;
    }

    void debugPrintFullTree(PrintWriter writer, boolean printIntervals) {
        this.curDepth = 0;
        this.preOrderPrint(writer, false, this.latestBranch.get(0));
        if (printIntervals) {
            writer.println("\nDetails of intervals:");
            this.curDepth = 0;
            this.preOrderPrint(writer, true, this.latestBranch.get(0));
        }
        writer.println('\n');
    }
}

