/*
 * @author Eyal Maderer
 */

package jdtcomments;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.IfStatement;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.PostfixExpression;
import org.eclipse.jdt.core.dom.PrefixExpression;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jdt.core.dom.WhileStatement;

import utils.CommentTypes;
import utils.MethodProfileData;
import utils.ObjectMarkerLocation;
import utils.ProfileDataHolder;
import utils.Variable;

// TODO: Auto-generated Javadoc
/**
 * The Class BaseCommentDetector.
 * 
 * @author Eyal Maderer
 * 
 *         This class was first written by Isaeed Mohanna, but did not have any
 *         logics besides the markers, I added the logics for traveling on the
 *         AST tree that are common to most of the detectors. This class is the
 *         base comment detector for all our detectors all of our detectors will
 *         extend this class for there specific implementations, this class
 *         holds all actions that are similar to all comments like initiating
 *         the process and MarkerCreator initialization. Most of the logics of
 *         the plug-in is located here. A lot of common visit and end visit
 *         methods can be found here due to the fact that most of the detectors
 *         need the same behavior from the different nodes.
 */
public abstract class BaseCommentDetector extends ASTVisitor 
{
	
	/** The _b inside block. */
	private boolean _bInsideBlock;
	
	/** The _b inside loop. */
	private boolean _bInsideLoop;
	
	/** The _col variables values. */
	private Collection<Variable> _colVariablesValues = new ArrayList<Variable>();
	
	/** The _s current assinged variable. */
	private String _sCurrentAssingedVariable;
	
	/** The _col objects locations. */
	private Collection<ObjectMarkerLocation> _colObjectsLocations = new ArrayList<ObjectMarkerLocation>();
	
	/** The _col members. */
	private Collection<VariableDeclarationFragment>_colMembers = new ArrayList<VariableDeclarationFragment>();
	
	/** The _col loop variables. */
	private Collection<String> _colLoopVariables = new ArrayList<String>();
	
	/** The _b currently in assigment. */
	private boolean _bCurrentlyInAssigment = false;
	
	/** The _i block count. */
	private int _iBlockCount = 0;
	
	/** The _i loop count. */
	private int _iLoopCount = 0;
	
	/** The _b inside critical function. */
	private boolean _bInsideCriticalFunction = false;
	
	/** The _b field decleration. */
	private boolean _bFieldDecleration = false;
	
	/** The _b is in condition. */
	private boolean _bIsInCondition = false;
	
	/** The _i condition counter. */
	private int _iConditionCounter = 0;
	
	// ============================= Protected data members
	// These members are accessible from all inherited class i.e. all comments

	/** The _icompilation unit. */
	private ICompilationUnit _icompilationUnit;
	/**
	 * Holds reference to the current working compilation unit. will be used for
	 * getting line numbers
	 */
	private CompilationUnit _compilationUnit;

	/** Reference to a marker creator which will be placing markers at the required locations. */
	private MarkerCreator _marker;

	// ============================= Public methods

	/**
	 * Starts the process of comment detecting on the supplied compilation unit.
	 * 
	 * @param unit
	 *            the AST root node. Bindings have to have been resolved.
	 * @param icoUnit
	 *            the ICompilationUnit unit
	 */
	public void process(CompilationUnit unit, ICompilationUnit icoUnit) {

		// save the compilation unit for later use
		setCompilationUnit(unit);
		
		setIcompilationUnit(icoUnit);

		// create a newer marker writer
		setMarker(new MarkerCreator(icoUnit));

		// start the AST Visitor
		unit.accept(this);

	}

	/**
	 * This function will be overridden in all inherited subclasses. it should
	 * calculate the line number then place the specific comment on that place
	 * 
	 * @param position
	 *            of placing the marker
	 * @param length
	 * 			length of code to highlight
	 */
	abstract protected void placeProblemMarkerSpecific(int position, int length);
	
	/**
	 * Place problem marker.
	 *
	 * @param position the position
	 * @param length the length
	 */
	protected void placeProblemMarker(int position, int length)
	{
		if(isInsideCriticalFunction())
		{
			placeProblemMarkerSpecific(position, length);
		}
	}
	
	/**
	 * Place book mark.
	 *
	 * @param position the position
	 * @param length the length
	 * @param marker the marker
	 * @param additionalCommant the additional commant
	 */
	protected void placeBookMark(int position, int length, MarkerCreator marker, String additionalCommant) 
	{
		marker.placeBookmark(position, length, CommentTypes.CRITICAL_METHOD, additionalCommant);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.TypeDeclaration)
	 */
	@Override
	public void endVisit(TypeDeclaration node)
	{
		getMembers().clear();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.FieldDeclaration)
	 */
	@Override
	public boolean visit(FieldDeclaration node)
	{
		try
		{
			TypeDeclaration cl = (TypeDeclaration) node.getParent();
			if(1 != cl.getFields().length)
			{
				_bFieldDecleration = true;
			}
		}
		catch (Exception e) {
			return false;
		}

		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.FieldDeclaration)
	 */
	@Override
	public void endVisit(FieldDeclaration node)
	{
		_bFieldDecleration = false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.IfStatement)
	 */
	@Override 
	public boolean visit(IfStatement node)
	{
		setIsInCondition(true);
		_iConditionCounter++;
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.IfStatement)
	 */
	public void endVisit(IfStatement node)
	{
		_iConditionCounter--;
		if(0 == _iConditionCounter)
		{
			setIsInCondition(false);
		}
	}
		
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.WhileStatement)
	 */
	@Override
	public boolean visit(WhileStatement node)
	{
		setInsideLoop(true);
		_iLoopCount++;
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.WhileStatement)
	 */
	@Override
	public void endVisit(WhileStatement node)
	{
		_iLoopCount--;
		if(_iLoopCount == 0)
		{
			setInsideLoop(false);
		}
	}

	/**
	 * The is a for statement, we want to add all the variables that change in
	 * this loop's declaration (i.e. x++) to the collection of all the loop's
	 * variables, update the loop's counter and set the flag that indicates that
	 * we are inside a loop to true.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(ForStatement node)
	{
		for(Object obj : node.updaters())
		{
			if(obj instanceof Expression)
			{
				if(obj instanceof PostfixExpression)
				{
					getLoopVariables().add(((PostfixExpression)obj).getOperand().toString());
				}

				if(obj instanceof PrefixExpression)
				{
					getLoopVariables().add(((PrefixExpression)obj).getOperand().toString());
				}
				
				if(obj instanceof Assignment)
				{
					getLoopVariables().add(((Assignment)obj).getLeftHandSide().toString());
				}
			}
		}
		_iLoopCount++;
		setInsideLoop(true);
		return true;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.ForStatement)
	 */
	@Override
	public void endVisit(ForStatement node)
	{
		getLoopVariables().clear();
		_iLoopCount--;
		if(_iLoopCount == 0)
		{
			setInsideLoop(false);
		}
	}

	/**
	 * Sets true to the flag that tells us that we are in a block and updates
	 * the block count.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(Block node) 
	{
		_iBlockCount++;
		setInsideBlock(true);		
		return true;
	}

	/**
	 * Calls the end block visit method.
	 * 
	 * @param node
	 *            the node
	 */
	@Override
	public void endVisit(Block node) 
	{
		endBlockVisit();
	}

	/**
	 * End block visit.
	 * We go over all the variables, and if a variable's scope has ended,
	 * we will delete him from the collection of all the variables.
	 */
	protected void endBlockVisit() 
	{
		Collection<Variable> colCurrentVariables = new ArrayList<Variable>();
		for(Variable var:getVariablesValues())
		{
			if(var.getBlockDeclared() != _iBlockCount)
			{
				colCurrentVariables.add(var);
			}
		}
		setVariablesValues(colCurrentVariables);
		_iBlockCount--;
		if(_iBlockCount==0)
		{
			setInsideBlock(false);
		}
	}

	/**
	 * this is a new variable, we want to add him to the variables collection or
	 * the members collection with the attributes needed.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(VariableDeclarationFragment node)
	{
		if(_bFieldDecleration)
		{
			getMembers().add(node);
		}
		String value = null;
		if(null != node.getInitializer())
		{
			setCurrentlyInAssigment(true);
		}
		if(null != node.getInitializer() && node.getInitializer() instanceof NumberLiteral)
		{
			value = node.getInitializer().toString();
		}
		else if(null != node.getInitializer() && node.getInitializer() instanceof SimpleName)
		{
			value = FindValue(node.getInitializer().toString(),  "=");
		}
		getVariablesValues().add(new Variable(node.getName().toString(), value, _iBlockCount));
		return true;
}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.VariableDeclarationFragment)
	 */
	@Override
	public void endVisit(VariableDeclarationFragment node)
	{
		setCurrentlyInAssigment(false);
	}
	
	/**
	 * This is an assignment node, we will set the current assigned variable to
	 * the variable on the left side of the expression(i.e. x=9, the current
	 * assigned variable will be 'x'. Then we will find this variable in the
	 * collection of all the variables and set the current block as the last
	 * block he has been changed in. if the value of the right hand side of the
	 * variable is not a literal or that we are in a condition, we don't know
	 * the value and we put there null for 'unknown' unless it's a simple name
	 * and in that case we will try to find the variable's value(the variable
	 * on the left of the assignment), else we will put the value
	 * of the right side of the expression. we also mark the scope of this
	 * change as block changed starts and block changed ends.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(Assignment node)
	{
		setCurrentAssingedVariable(node.getLeftHandSide().toString());
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(node.getLeftHandSide().toString()) && isCurrentInstance(var))
			{
				var.setBlockChange(_iBlockCount);
				var.setBlockChangedEnds(getBlockEndPoint(node));
				var.setBlockChangedStarts(getBlockStartPoint(node));
				if(node.getRightHandSide() instanceof NumberLiteral && !isInCondition() && node.getOperator().toString().equalsIgnoreCase("="))
				{
					var.setCurrentValue(node.getRightHandSide().toString());
				}
				else if(node.getRightHandSide() instanceof SimpleName && !isInCondition())
				{
					var.setCurrentValue(FindValue(node.getRightHandSide().toString(),node.getOperator().toString()));
				}
				else 
				{
					var.setCurrentValue(null);
				}

			}
		}
		setCurrentlyInAssigment(true);
		return true;
	}
	
	/**
	 * Find value.
	 * 
	 * @param rightHandVariable
	 *            the right hand variable
	 * @param operator
	 *            - the assignment operator
	 * @return the current value of the Left hand variable in the assignment
	 */
	private String FindValue(String rightHandVariable, String operator) 
	{
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(rightHandVariable) && isCurrentInstance(var))
			{
				if(operator.equalsIgnoreCase("="))
				{
					return var.getCurrentValue();
				}
			}
		}
		return null;
	}

	/**
	 * Gets the block end point.
	 * 
	 * @param node
	 *            the node
	 * @return the last position of the block of the current node
	 */
	private int getBlockEndPoint(ASTNode node) 
	{
		if(node instanceof Block)
		{
			return ((Block)node).getStartPosition() + ((Block)node).getLength();
		}
		return getBlockEndPoint(node.getParent()) ;
	}
	
	/**
	 * Gets the block start point.
	 * 
	 * @param node
	 *            the node
	 * @return the start position of the block of the current node
	 */
	private int getBlockStartPoint(ASTNode node) 
	{
		if(node instanceof Block)
		{
			return ((Block)node).getStartPosition();
		}
		return getBlockStartPoint(node.getParent()) ;
	}

	/**
	 * sets false to the flag that tells us if we are in an assignment.
	 * 
	 * @param node
	 *            the node
	 */
	@Override
	public void endVisit(Assignment node)
	{
		setCurrentlyInAssigment(false);
	}
	
	/**
	 * This is a post fix expression (i.e. x++), we need to find the variable in
	 * the collection of all the variables, change the block he was changed in
	 * and to try to calculate the new value.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(PostfixExpression node)
	{
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(node.getOperand().toString()))
			{
				if(node.getParent() instanceof ForStatement)
				{
					var.setBlockChange(_iBlockCount + 1);
				}
				else
				{
					var.setBlockChange(_iBlockCount);
				}
				var.setBlockChangedEnds(getBlockEndPoint(node));
				var.setBlockChangedStarts(getBlockStartPoint(node));
				if(null != var.getCurrentValue())
				{
					if(node.getOperator().toString().equalsIgnoreCase("++"))
					{
						var.setCurrentValue(String.valueOf(Double.valueOf(var.getCurrentValue()) + 1));
					}
					else if (node.getOperator().toString().equalsIgnoreCase("--"))
					{
						var.setCurrentValue(String.valueOf(Double.valueOf(var.getCurrentValue()) - 1));
					}
					if(isInsideLoop())
					{
						var.setCurrentValue(null);
					}
				}
			}
		}
		return true;
	}
	
	/**
	 * This is a post fix expression (i.e. ++x), we need to find the variable in
	 * the collection of all the variables, change the block he was changed in
	 * and to try to calculate the new value.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(PrefixExpression node)
	{
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(node.getOperand().toString()))
			{
				var.setBlockChange(_iBlockCount);
				var.setBlockChangedEnds(getBlockEndPoint(node));
				var.setBlockChangedStarts(getBlockStartPoint(node));
				if(null != var.getCurrentValue())
				{		
					if(node.getOperator().toString().equalsIgnoreCase("++"))
					{
						var.setCurrentValue(String.valueOf(Double.valueOf(var.getCurrentValue()) + 1));
					}
					else if (node.getOperator().toString().equalsIgnoreCase("--"))
					{
						var.setCurrentValue(String.valueOf(Double.valueOf(var.getCurrentValue()) - 1));
					}
					if(isInsideLoop())
					{
						var.setCurrentValue(null);
					}
				}
			}
			
		}
		return true;
	}
	
	/**
	 * We need to check if we are in an assignment right now, if not the infix
	 * expression does not interest us and we return false.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(InfixExpression node)
	{
		if(!isCurrentlyInAssigment())
		{
			return false;
		}
	
		return true;
	}

	/**
	 * If we are in show comments only in critical methods mode, and this is a
	 * critical method than we should set the flag of is inside critical method
	 * to true.
	 * 
	 * @param node
	 *            the node
	 */
	@Override
	public void endVisit(MethodDeclaration node)
	{
		if(!ProfileDataHolder.getInstance().isRunOnlyCriticalCode())
		{
			setInsideCriticalFunction(true);
		}
		else
		{
			setInsideCriticalFunction(false);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.SingleVariableDeclaration)
	 */
	public boolean visit(SingleVariableDeclaration node)
	{
		getVariablesValues().add(new Variable(node.getName().toString(), null, _iBlockCount+1));
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@Override
	public boolean visit(MethodDeclaration node)
	{
		if(!ProfileDataHolder.getInstance().isRunOnlyCriticalCode())
		{
			setInsideCriticalFunction(true);
			return true;
		}
		String sPackage = null;
		String sFunction = node.getName().toString();
		TypeDeclaration parent = (TypeDeclaration) node.getParent();
		String sClass = parent.getName().toString();
		CompilationUnit unit = (CompilationUnit) parent.getParent();
		try
		{
			sPackage = unit.getPackage().toString();
			sPackage = sPackage.substring(sPackage.indexOf("package") + 8);
			sPackage = sPackage.substring(0, sPackage.length()-2);
		}
		catch (Exception e)
		{
			sPackage = "";
		}
		boolean bMark = false;
		MethodProfileData methodData = null;
		for(MethodProfileData data:ProfileDataHolder.getInstance().getCriticalMethods())
		{
			if(sPackage.trim().equalsIgnoreCase(data.getPackage()) && sClass.trim().equalsIgnoreCase(data.getClassName())
					&& sFunction.trim().equalsIgnoreCase(data.getFunction()))
			{
				methodData = data;
				bMark = true;
			}
		}
		if(bMark)
		{
			setInsideCriticalFunction(true);
		}
		if(!bMark ||(methodData != null && methodData.isWasMarked()))
		{
			return true;
		}
		String sMessage = "It was called " + methodData.getCount() + " times and \n " +
		"it ran for " + methodData.getTime() + " miliseconds"
		+ " which is " + methodData.getPct() + "% of the total run.";
		placeBookMark(node.getName().getStartPosition(),  node.getName().toString().length(), getMarker(), sMessage);
		methodData.setWasMarked(true);
		return true;
	}

	
	/**
	 * Checks if is current instance.
	 *
	 * @param checkedVar the checked var
	 * @return true, if is current instance
	 */
	protected boolean isCurrentInstance(Variable checkedVar)
	{
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(checkedVar.getName()) && var.getBlockDeclared() > checkedVar.getBlockDeclared())
			{
				return false;
			}
		}
		return true;
	}


	/**
	 * Sets the current assinged variable.
	 *
	 * @param _sCurrentAssingedVariable the new current assinged variable
	 */
	public void setCurrentAssingedVariable(String _sCurrentAssingedVariable) 
	{
		this._sCurrentAssingedVariable = _sCurrentAssingedVariable;
	}

	/**
	 * Gets the current assinged variable.
	 *
	 * @return the current assinged variable
	 */
	public String getCurrentAssingedVariable() 
	{
		return _sCurrentAssingedVariable;
	}

	/**
	 * Sets the objects locations.
	 *
	 * @param _colObjectsLocations the new objects locations
	 */
	public void setObjectsLocations(Collection<ObjectMarkerLocation> _colObjectsLocations) 
	{
		this._colObjectsLocations = _colObjectsLocations;
	}

	/**
	 * Gets the objects locations.
	 *
	 * @return the objects locations
	 */
	public Collection<ObjectMarkerLocation> getObjectsLocations() 
	{
		return _colObjectsLocations;
	}

	/**
	 * Sets the currently in assigment.
	 *
	 * @param _bCurrentlyInAssigment the new currently in assigment
	 */
	public void setCurrentlyInAssigment(boolean _bCurrentlyInAssigment) 
	{
		this._bCurrentlyInAssigment = _bCurrentlyInAssigment;
	}

	/**
	 * Checks if is currently in assigment.
	 *
	 * @return true, if is currently in assigment
	 */
	public boolean isCurrentlyInAssigment()
	{
		return _bCurrentlyInAssigment;
	}

	/**
	 * Sets the block count.
	 *
	 * @param _iBlockCoubt the new block count
	 */
	public void setBlockCount(int _iBlockCoubt) 
	{
		this._iBlockCount = _iBlockCoubt;
	}

	/**
	 * Gets the block count.
	 *
	 * @return the block count
	 */
	public int getBlockCount()
	{
		return _iBlockCount;
	}
	

	/**
	 * Sets the inside block.
	 *
	 * @param _bInsideBlock the new inside block
	 */
	public void setInsideBlock(boolean _bInsideBlock)
	{
		this._bInsideBlock = _bInsideBlock;
	}

	/**
	 * Checks if is inside block.
	 *
	 * @return true, if is inside block
	 */
	public boolean isInsideBlock() 
	{
		return _bInsideBlock;
	}

	/**
	 * Sets the inside loop.
	 *
	 * @param _bInsideLoop the new inside loop
	 */
	public void setInsideLoop(boolean _bInsideLoop) 
	{
		this._bInsideLoop = _bInsideLoop;
	}

	/**
	 * Checks if is inside loop.
	 *
	 * @return true, if is inside loop
	 */
	public boolean isInsideLoop()
	{
		return _bInsideLoop;
	}

	/**
	 * Sets the variables values.
	 *
	 * @param _colVariablesValues the new variables values
	 */
	public void setVariablesValues(Collection<Variable> _colVariablesValues)
	{
		this._colVariablesValues = _colVariablesValues;
	}

	/**
	 * Gets the variables values.
	 *
	 * @return the variables values
	 */
	public Collection<Variable> getVariablesValues()
	{
		return _colVariablesValues;
	}

	/**
	 * Sets the loop count.
	 *
	 * @param _iLoopCount the new loop count
	 */
	public void setLoopCount(int _iLoopCount)
	{
		this._iLoopCount = _iLoopCount;
	}

	/**
	 * Gets the loop count.
	 *
	 * @return the loop count
	 */
	public int getLoopCount() 
	{
		return _iLoopCount;
	}

	/**
	 * Sets the marker.
	 *
	 * @param _marker the new marker
	 */
	public void setMarker(MarkerCreator _marker) 
	{
		this._marker = _marker;
	}

	/**
	 * Gets the marker.
	 *
	 * @return the marker
	 */
	public MarkerCreator getMarker() 
	{
		return _marker;
	}

	/**
	 * Sets the compilation unit.
	 *
	 * @param _compilationUnit the new compilation unit
	 */
	public void setCompilationUnit(CompilationUnit _compilationUnit) 
	{
		this._compilationUnit = _compilationUnit;
	}

	/**
	 * Gets the compilation unit.
	 *
	 * @return the compilation unit
	 */
	public CompilationUnit getCompilationUnit() 
	{
		return _compilationUnit;
	}

	/**
	 * Sets the icompilation unit.
	 *
	 * @param icompilationUnit the new icompilation unit
	 */
	public void setIcompilationUnit(ICompilationUnit icompilationUnit)
	{
		_icompilationUnit = icompilationUnit;
	}

	/**
	 * Gets the icompilation unit.
	 *
	 * @return the icompilation unit
	 */
	public ICompilationUnit getIcompilationUnit() 
	{
		return _icompilationUnit;
	}

	/**
	 * Sets the inside critical function.
	 *
	 * @param bInsideCriticalFunction the new inside critical function
	 */
	public void setInsideCriticalFunction(boolean bInsideCriticalFunction) 
	{
		_bInsideCriticalFunction = bInsideCriticalFunction;
	}

	/**
	 * Checks if is inside critical function.
	 *
	 * @return true, if is inside critical function
	 */
	public boolean isInsideCriticalFunction()
	{
		return _bInsideCriticalFunction;
	}

	/**
	 * Sets the members.
	 *
	 * @param colMembers the new members
	 */
	public void setMembers(Collection<VariableDeclarationFragment> colMembers) 
	{
		_colMembers = colMembers;
	}

	/**
	 * Gets the members.
	 *
	 * @return the members
	 */
	public Collection<VariableDeclarationFragment> getMembers() 
	{
		return _colMembers;
	}

	/**
	 * Sets the loop variables.
	 *
	 * @param colLoopVariables the new loop variables
	 */
	public void setLoopVariables(Collection<String> colLoopVariables)
	{
		_colLoopVariables = colLoopVariables;
	}

	/**
	 * Gets the loop variables.
	 *
	 * @return the loop variables
	 */
	public Collection<String> getLoopVariables()
	{
		return _colLoopVariables;
	}

	/**
	 * Sets the checks if is in condition.
	 *
	 * @param bIsInCondition the new checks if is in condition
	 */
	public void setIsInCondition(boolean bIsInCondition)
	{
		_bIsInCondition = bIsInCondition;
	}

	/**
	 * Checks if is in condition.
	 *
	 * @return true, if is in condition
	 */
	public boolean isInCondition() 
	{
		return _bIsInCondition;
	}


}

