/*
 * 
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.ParenthesizedExpression;
import org.eclipse.jdt.core.dom.WhileStatement;

import utils.ObjectMarkerLocation;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * The Class CommonSubExpressionDetector.
 * 
 * @author Eyal Maderer
 * 
 *         <p>
 *         The Class CommonSubExpressionDetector.
 *         <p>
 *         <p>
 *         In our implementation of ObjectInliningDetector.java we extended
 *         our base comment detector and try to detect all the expressions that
 *         appear inside loops and determine if an elimination of the expression
 *         from the loop is recommended. We hold a collection of all the
 *         expressions we encounter inside loops we check if the expressions
 *         appear more than once inside loops and if so we put a proper comment
 *         each time this expression appears.
 *         <p>
 */
public class CommonSubExpressionDetector extends BaseCommentDetector 
{
	
	/** The _mp expressions. */
	private Map<Expression, ObjectMarkerLocation> _mpExpressions = new HashMap<Expression, ObjectMarkerLocation>();
	
	/** The _col markers. */
	private Collection<ObjectMarkerLocation> _colMarkers = new ArrayList<ObjectMarkerLocation>();

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarkerSpecific(int, int)
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) 
	{
		getMarker().placeProblem(position, length, CommentTypes.COMMON_SUB_EXPRESSION);
	}

	/**
	 * Place problem marker.
	 *
	 * @param position the position	
	 * @param length the length
	 * @param marker the marker
	 * @param expression the expression
	 */
	protected void placeProblemMarker(int position, int length, MarkerCreator marker, String expression) 
	{
		if(isInsideCriticalFunction())
		{
			marker.placeProblem(position, length, CommentTypes.COMMON_SUB_EXPRESSION, expression);
		}
	}


	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.ForStatement)
	 */
	@Override
	public void endVisit(ForStatement node)
	{
		super.endVisit(node);
		endLoopVisit();
	}

	/**
	 * End loop visit.
	 */
	private void endLoopVisit() 
	{
		for(ObjectMarkerLocation classLocation:getMarkers())
		{
			placeProblemMarker(classLocation.getStartLocation(), classLocation.getLength(), classLocation.getMarker(), classLocation.getObjectName());
		}
		getMarkers().clear();
		getExpressions().clear();
		setInsideLoop(false);
	}

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.WhileStatement)
	 */
	@Override
	public void endVisit(WhileStatement node)
	{
		super.endVisit(node);
		endLoopVisit();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.ParenthesizedExpression)
	 */
	@Override
	public boolean visit(ParenthesizedExpression node)
	{
		Expression currentExpression = node.getExpression();
		return parseExpression(node, currentExpression);
	}
	
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.Assignment)
	 */
	@Override
	public boolean visit(Assignment node)
	{
		Expression currentExpression = node.getRightHandSide();
		return parseExpression(node.getRightHandSide(), currentExpression);
	}

	/**
	 * Parses the expression.
	 *
	 * @param node the node
	 * @param currentExpression the current expression
	 * @return true, if successful
	 */
	private boolean parseExpression(ASTNode node, Expression currentExpression)
	{
		if(currentExpression.getNodeType() == ASTNode.NUMBER_LITERAL)
		{
			return true;
		}
		if(currentExpression instanceof InfixExpression)
		{
			if(((InfixExpression)currentExpression).hasExtendedOperands())
			{
				for(Object obj:((InfixExpression)currentExpression).extendedOperands())
				{
					if(obj instanceof InfixExpression)
					{
						parseExpression((ASTNode)obj,(InfixExpression)obj);
					}
					if(obj instanceof ParenthesizedExpression)
					{
						visit((ParenthesizedExpression)obj);
					}
				}
				
			}
			if(((InfixExpression)currentExpression).getLeftOperand() instanceof ParenthesizedExpression)
			{
				parseExpression(((InfixExpression)currentExpression).getLeftOperand(), ((InfixExpression)currentExpression).getLeftOperand());
			}
			if(((InfixExpression)currentExpression).getRightOperand() instanceof ParenthesizedExpression)
			{
				parseExpression(((InfixExpression)currentExpression).getRightOperand(), ((InfixExpression)currentExpression).getRightOperand());
			}
		}
		if(isInsideLoop())
		{		
			for(Expression exp:getExpressions().keySet())
			{
				if(exp.toString().equalsIgnoreCase(currentExpression.toString()))
				{
					ObjectMarkerLocation classLoaction = new ObjectMarkerLocation(exp.toString(), node.getStartPosition(),node.getLength(), getMarker());
					getMarkers().add(classLoaction);
					getMarkers().add(getExpressions().get(exp));
					return true;
				}
				
			}
			getExpressions().put(currentExpression, new ObjectMarkerLocation(currentExpression.toString(), node.getStartPosition(),node.getLength(), getMarker()));
			
			return true;
		}
		return false;
	}	
	

	/**
	 * Sets the expressions.
	 *
	 * @param mpExpressions the mp expressions
	 */
	public void setExpressions(Map<Expression, ObjectMarkerLocation> mpExpressions) 
	{
		this._mpExpressions = mpExpressions;
	}

	/**
	 * Gets the expressions.
	 *
	 * @return the expressions
	 */
	public Map<Expression, ObjectMarkerLocation> getExpressions()
	{
		return _mpExpressions;
	}

	/**
	 * Sets the markers.
	 *
	 * @param _colMarkers the new markers
	 */
	public void setMarkers(Collection<ObjectMarkerLocation> _colMarkers) 
	{
		this._colMarkers = _colMarkers;
	}

	/**
	 * Gets the markers.
	 *
	 * @return the markers
	 */
	public Collection<ObjectMarkerLocation> getMarkers()
	{
		return _colMarkers;
	}

}
