/*
 * 
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.SimpleName;

import utils.CommentTypes;
import utils.ObjectMarkerLocation;
import utils.Variable;

// TODO: Auto-generated Javadoc
/**
 * The Class ConstantPropagationDetector.
 * 
 * @author Eyal Maderer
 * 
 *         <p>
 *         The Class ConstantPropagationDetector.
 *         <p>
 *         <p>
 *         * In our implementation of CopyPropagationDetector.java and
 *         ConstantPropagationDetector.java we extended our base comment
 *         detector. The implementation of copy propagation detector is very
 *         simple, we just need to go over all the Assignment and check if the
 *         right left operand is the same as the right operand. The
 *         implementation of constant propagation is more complicated, we hold a
 *         collection of the entire variable in the code, the scope level where
 *         they were made, their initial values, their current value, and the
 *         scope where they were last changed. Than we go over all the loops and
 *         see if there is an assignment inside the loop which the right hand
 *         operand is a variable that his value is known to us and does not
 *         change in the loops scope, if so this variable can be replaced with
 *         a constant which is the last value he was assigned to, if the value
 *         of the variable is unknown to us, or if this variables value changes
 *         within the loop, constant propagation cannot be implemented there.
 *         <p>
 */
public class ConstantPropagationDetector extends BaseCommentDetector 
{

	
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarkerSpecific(int, int)
	 */
	protected void placeProblemMarkerSpecific(int position, int length) 
	{
		getMarker().placeProblem(position, length, CommentTypes.CONSTANT_PROPAGATION);
	}
	
	/**
	 * Place problem marker.
	 *
	 * @param position the position
	 * @param length the length
	 * @param marker the marker
	 * @param additionalComment the additional comment
	 */
	protected void placeProblemMarker(int position, int length, MarkerCreator marker, String additionalComment) 
	{
		if(isInsideCriticalFunction())
		{
			marker.placeProblem(position, length, CommentTypes.CONSTANT_PROPAGATION, additionalComment);
		}
	}


	/**
	 * Here we check if we are in assignment and if so we check if the variable
	 * of the node is not the variable on the left side of the assignment and if
	 * he is not one of the current loop's variable (i.e. i++). if so, and we
	 * are indeed inside a loop we will mark this location as a suspect for
	 * constant propagation, when the current block ends, we will see if this
	 * variable's value remained a constant all this block's scope and if so we
	 * will place a comment there.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(SimpleName node)
	{
		if(!isCurrentlyInAssigment() || node.toString().equalsIgnoreCase(getCurrentAssingedVariable()))
		{
			return false;
		}
		for(String field:getLoopVariables())
		{
			if(field.equalsIgnoreCase(node.toString()))
			{
				return false;
			}
		}
		if(isInsideLoop())
		{
			ObjectMarkerLocation objectLoaction = new ObjectMarkerLocation(node.toString(), node.getStartPosition(),node.getLength(), getMarker());
			//Set the block where this location is
			objectLoaction.setBlockCount(getBlockCount());
			getObjectsLocations().add(objectLoaction);
		}
		return true;
	}

	/**
	 * The block ended, we will go all over the locations we marked as suspects
	 * and check if they are not already marked, if so, we will check all the
	 * variables and see if we can find a variable with the same name as the
	 * name of the variable we marked as a suspect in the location, we need to
	 * check if this is the current instance of the variable (for example we can
	 * have 2 variables named 'x' and we need to make sure that the 'x' we are
	 * now seeing is really the 'x' we are looking for). if so and the value of
	 * the variable is a constant in the block we will place the comment.
	 * 
	 * @param node
	 *            the node
	 */
	@Override
	public void endVisit(Block node) 
	{
		for(ObjectMarkerLocation obj:getObjectsLocations())
		{
			if(obj.isMarked())
			{
				continue;
			}
			for(Variable var:getVariablesValues())
			{
				//Check that this is the variable and that he was not changed in this scope and that we are inside a loop
				if(obj.getObjectName().equalsIgnoreCase(var.getName()) && (var.getBlockChange() < getBlockCount()|| 
						!isChangedInTheCurrentScope(node, var)) && isInsideLoop())
				{
					if(isCurrentInstance(var))
					{
						if(null == var.getCurrentValue())
						{
							obj.setMarked(true);
						}
						else
						{
							obj.setMarked(true);
							placeProblemMarker(obj.getStartLocation(), obj.getLength(), obj.getMarker(), var.getCurrentValue());
						}
					}
				}
			}
		}
		
		//Clean all the suspects that their scope has just ended
		Collection<ObjectMarkerLocation> colObjects = new ArrayList<ObjectMarkerLocation>();
		for(ObjectMarkerLocation obj:getObjectsLocations())
		{
			if(obj.getBlockCount()!= getBlockCount())
			{
				colObjects.add(obj);
			}
		}
		setObjectsLocations(colObjects);
		endBlockVisit();
	}

	/**
	 * Checks if is changed in the current scope.
	 * 
	 * @param node
	 *            the node
	 * @param var
	 *            the var
	 * @return true if the variable has been last changed in the current scope
	 */
	private boolean isChangedInTheCurrentScope(Block node, Variable var)
	{
		return (var.getBlockChangedEnds() < (node.getStartPosition()+node.getLength())) &&
		(var.getBlockChangedStarts() > node.getStartPosition());
	}

	
}
