/*
 * 
 */
package jdtcomments;

import org.eclipse.jdt.core.dom.Assignment;

import utils.CommentTypes;
import utils.ObjectMarkerLocation;
import utils.Variable;

// TODO: Auto-generated Javadoc
/**
 * The Class CopyPropagationDetector.
 * 
 * @author Eyal Maderer
 * 
 *         <p>
 *         The Class CopyPropagationDetector.
 *         <p>
 *         <p>
 *         In our implementation of CopyPropagationDetector.java and
 *         ConstantPropagationDetector.java we extended our base comment
 *         detector. The implementation of copy propagation detector is very
 *         simple, we just need to go over all the Assignment and check if the
 *         right left operand is the same as the right operand. The
 *         implementation of constant propagation is more complicated, we hold a
 *         collection of the entire variable in the code, the scope level where
 *         they were made, their initial values, their current value, and the
 *         scope where they were last changed. Than we go over all the loops and
 *         see if there is an assignment inside the loop which the right hand
 *         operand is a variable that his value is known to us and does not
 *         change in the loops scope, if so this variable can be replaced with
 *         a constant which is the last value he was assigned to, if the value
 *         of the variable is unknown to us, or if this variables value changes
 *         within the loop, constant propagation cannot be implemented there.
 *         <p>
 */
public class CopyPropagationDetector extends BaseCommentDetector 
{
	
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarkerSpecific(int, int)
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) 
	{
		getMarker().placeProblem(position, length, CommentTypes.COPY_PROPAGATION);
	}
	
	/**
	 * Place problem marker.
	 *
	 * @param position the position
	 * @param length the length
	 * @param marker the marker
	 */
	protected void placeProblemMarker(int position, int length, MarkerCreator marker) 
	{
		if(isInsideCriticalFunction())
		{
			marker.placeProblem(position, length, CommentTypes.COPY_PROPAGATION);
		}
	}
	
	/**
	 * We want to check if a variable is being assigned to himself or if a
	 * Variable is being assigned the same value as he did before.
	 * 
	 * @param node
	 *            the node
	 * @return true, if successful
	 */
	@Override
	public boolean visit(Assignment node)
	{
		super.visit(node);
		if(!node.getOperator().toString().equalsIgnoreCase("="))
		{
			return true;
		}
		if(node.getLeftHandSide().toString().equalsIgnoreCase(node.getRightHandSide().toString())
				|| checkForSameValues(node.getLeftHandSide().toString(), node.getRightHandSide().toString()))
		{
			ObjectMarkerLocation classLoaction = new ObjectMarkerLocation(node.toString(), node.getStartPosition(),node.getLength(), getMarker());
			getObjectsLocations().add(classLoaction);
		}
		return true;
	}
	
	/**
	 * Check for same values.
	 * 
	 * @param leftOperand
	 *            the left operand
	 * @param rightOperand
	 *            the right operand
	 * @return true if the 2 operand's values are the same
	 */
	private boolean checkForSameValues(String leftOperand, String rightOperand) 
	{
		Variable vLeft = null;
		Variable vRight = null;
		for(Variable var:getVariablesValues())
		{
			if(var.getName().equalsIgnoreCase(leftOperand) && isCurrentInstance(var))
			{
				vLeft = var;
			}
			if(var.getName().equalsIgnoreCase(rightOperand) && isCurrentInstance(var))
			{
				vRight = var;
			}
			
		}
		if(null != vLeft && null != vRight && null != vLeft.getCurrentValue() && null != vRight.getCurrentValue())
		{
			if(vLeft.getCurrentValue().equalsIgnoreCase(vRight.getCurrentValue()))
			{
				return true;
			}
		}
		return false;
	}
	
	/**
	 * We are at the end of the assignment node, we can put the markers and
	 * clear the object's locations.
	 * 
	 * @param node
	 *            the node
	 */
	@Override
	public void endVisit(Assignment node)
	{
		for(ObjectMarkerLocation classLocation:getObjectsLocations())
		{
			placeProblemMarker(classLocation.getStartLocation(), classLocation.getLength(), classLocation.getMarker());
		}
		getObjectsLocations().clear();
	}

	
}
