/**
 * @author Isaeed Mohanna
 * May 28, 2009 2009
 * 10:22:58 AM
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.dom.ArrayAccess;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.BooleanLiteral;
import org.eclipse.jdt.core.dom.CharacterLiteral;
import org.eclipse.jdt.core.dom.DoStatement;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.NullLiteral;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.WhileStatement;

import utils.BlockInvariantVariablesDetector;
import utils.BlockVariantVariablesDetector;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * This class is responsible for detecting manual array initialization instead
 * of Array.Fill usage. the detector will detect the array initialization by the
 * following steps: 1) Visit Loops and mark them 2) find the loop iterator from
 * the expression 3) Visit Assignment statement and check if the left hand
 * operand is an array access node. 4) check if array access node index is a
 * variable event for nested array access.
 * 
 */
public class InitArrayDetector extends BaseCommentDetector {

	// ============================= Private data members

	/**
	 * This methods tells the nested number of loops at our given location. a
	 * counter is used instead of a regular boolean flag in order to detect
	 * nested loops
	 */
	int _loopCounter = 0;
	
	/** A list of all loop iterator. */
	ArrayList<String> _loopIterator = new ArrayList<String>();
	
	/** Holds a set of invariant variables inside the current block, it is filled upon first entrance to a loop. */
	Set<String> _blockInvariantVariables;

	// ============================= Overridden functions
	
	/* Add a problem marker and highlight the text
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarker(int, int)
	 */
	/**
	 * Place problem marker specific.
	 *
	 * @param position position to place the marker
	 * @param length length of code to highlight
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) {
		// Place Marker
		getMarker().placeProblem(position, length, CommentTypes.INEFFICIENT_CODING);
		
	}

	// ============================= Overridden Visitors
	// ============== For Statement
	/*
	 * Leaves a for loop tree leaf (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .ForStatement)
	 */
	@Override
	public void endVisit(ForStatement node) {
		// Decrease the loop counter
		_loopCounter--;

		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			// clear loop iterators list
			_loopIterator.clear();
			// Clear invariant variables list
			_blockInvariantVariables.clear();
		}

		super.endVisit(node);
	}

	/*
	 * Handles the entrance to a for statement tree (non-Javadoc)
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * ForStatement)
	 */
	@Override
	public boolean visit(ForStatement node) {

		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			_loopIterator.clear();
			// build invariant code set from block at first loop
			findBlockInvariantVariables(node);
		}
		// Increase the loop counter
		_loopCounter++;

		// find loop iterators candidates from expression
		findLoopExpressionVariables(node.getExpression());
		// add variables from for updater list
		// remove all untouched variables from loopIterator
		findLoopIterator(node);

		return true;
	}

	// ============== While statements

	/*
	 * leaves a while loop node(non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .WhileStatement)
	 */
	@Override
	public void endVisit(WhileStatement node) {
		// Decrease the loop counter
		_loopCounter--;

		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			// clear loop iterators list
			_loopIterator.clear();
			// Clear invariant variables list
			_blockInvariantVariables.clear();
		}
		super.endVisit(node);
	}

	/*
	 * handles entrance to a while statement node(non-Javadoc)
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * WhileStatement)
	 */
	@Override
	public boolean visit(WhileStatement node) {
		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			_loopIterator.clear();
			// build invariant code set from block at first loop
			findBlockInvariantVariables(node);
		}
		// Increase the loop counter
		_loopCounter++;

		// find loop iterators candidates from expression
		findLoopExpressionVariables(node.getExpression());
		// remove all untouched variables from loopIterator
		Block block = (Block) node.getBody();
		findLoopIterator(block);

		return true;
	}

	// ============== Do Statements

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .DoStatement)
	 */
	@Override
	public void endVisit(DoStatement node) {
		// Decrease the loop counter
		_loopCounter--;

		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			// clear loop iterators list
			_loopIterator.clear();
			// Clear invariant variables list
			_blockInvariantVariables.clear();
		}
		super.endVisit(node);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * DoStatement)
	 */
	@Override
	public boolean visit(DoStatement node) {
		// clear the variables map if this is a first level loop
		if (_loopCounter == 0) {
			_loopIterator.clear();
			// build invariant code set from block at first loop
			findBlockInvariantVariables(node);
		}
		// Increase the loop counter
		_loopCounter++;

		// find loop iterators candidates from expression
		findLoopExpressionVariables(node.getExpression());
		// remove all untouched variables from loopIterator
		Block block = (Block) node.getBody();
		findLoopIterator(block);

		return true;
	}

	// ============== Assignment

	/*
	 * Visits the assignment nodes (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment
	 * )
	 */
	@Override
	public boolean visit(Assignment node) {
		// check if we are inside a loop
		if (_loopCounter > 0) {
			if (node.getLeftHandSide() instanceof ArrayAccess) {
				// get the array access node
				ArrayAccess arrayAccess = (ArrayAccess) node.getLeftHandSide();
				// get array indexes and store them in array list
				ArrayList<String> indexs = new ArrayList<String>();

				if (getArrayAccessIndexs(arrayAccess, indexs)) {

					// check if right hand operand is a constant value, that is
					// a literal
					Expression tmp = node.getRightHandSide();

					if (isInvariant(tmp)) {

						// Run over all indexes and check if they appear in loop
						// iterators list
						boolean result = false;
						for (Iterator<String> iterator = indexs.iterator(); iterator.hasNext();) {
							String index = (String) iterator.next();

							// check if index inside loop iterators map
							if (_loopIterator.contains(index)) {
								result = true;
							}

						}
						if (result) {
							placeProblemMarker(node.getStartPosition(),node.getLength());
						}
					}
				}
			}
		}
		// check if we are assigning to an array
		return super.visit(node);
	}

	// ============================= Private Help methods
	/**
	 * Checks if the array access is of simple name type i.e. it checks if all
	 * indexes of the array access are of variables
	 *
	 * @param arrayAccess the array access
	 * @param indexs array list for storing all array indexs
	 * @return TRUE - Array Access with variable index FALSE - array access with
	 * at least one hard coded index
	 */
	private boolean getArrayAccessIndexs(ArrayAccess arrayAccess, ArrayList<String> indexs) {
		// Result holder
		boolean result = true;
		// check if this is a nested array access
		if (arrayAccess.getArray() instanceof ArrayAccess) {
			result = getArrayAccessIndexs((ArrayAccess) arrayAccess.getArray(), indexs);
		}
		// checks if current index is a variable
		if (!result) {
			return false;
		} else {
			// check if index is a variable
			if (arrayAccess.getIndex() instanceof SimpleName) {
				SimpleName sName = (SimpleName) arrayAccess.getIndex();
				indexs.add(sName.getIdentifier());
				return true;
			}
			// check if index is a infix expression
			if (arrayAccess.getIndex() instanceof InfixExpression) {
				InfixExpression iExpression = (InfixExpression) arrayAccess.getIndex();
				// handle infix expression inside array access
				findVariables(iExpression, indexs);
				return true;
			}

			return false;
		}
	}

	/**
	 * Find variables.
	 *
	 * @param exp the exp
	 * @param indexs - array list for saving variables from expression.
	 * @return boolean
	 */
	@SuppressWarnings("unchecked")
	private void findVariables(Expression exp, ArrayList<String> indexs) {
		// check if this an infix expression
		if (exp instanceof InfixExpression) {
			InfixExpression iExpression = (InfixExpression) exp;
			// check left hand
			findVariables(iExpression.getLeftOperand(), indexs);
			// check right operand
			findVariables(iExpression.getRightOperand(), indexs);
			// rotate over all extended operands and check them
			List extended = iExpression.extendedOperands();
			for (Iterator iterator = extended.iterator(); iterator.hasNext();) {
				Expression curExp = (Expression) iterator.next();
				// find variables
				findVariables(curExp, indexs);
			}
		}
		if (exp instanceof SimpleName) {
			// cast
			SimpleName sName = (SimpleName) exp;
			// add identifier to array indexes
			indexs.add(sName.getIdentifier());
		}

	}

	/**
	 * Check if the right side of the assignment is a constant value that is
	 * some literal type.
	 *
	 * @param exp Expression
	 * @return TRUE - A constant Value FALSE - Variable or expression
	 */
	private boolean isConstantValue(Expression exp) {
		// check for null
		if (exp == null) {
			return false;
		}
		// check for different types of literals
		if (exp instanceof NumberLiteral || exp instanceof StringLiteral
				|| exp instanceof BooleanLiteral || exp instanceof NullLiteral
				|| exp instanceof CharacterLiteral) {
			// if it is a literal type
			return true;
		}
		// none of these
		return false;
	}

	/**
	 * This methods finds all variable relevant to the loop decision. i.e. it
	 * adds all variables that decide wither the loop continues or halts that is
	 * the loop iterators. All found iterators are added to iterators array list
	 * 
	 * @param loopExp
	 *            - expression of the loop
	 */
	@SuppressWarnings("unchecked")
	private void findLoopExpressionVariables(Expression loopExp) {

		// Make sure that expression is of type infix expression
		if (loopExp instanceof InfixExpression) {
			// Cast infix expression
			InfixExpression infixExpression = (InfixExpression) loopExp;

			// deal with left hand operand
			findLoopExpressionVariables(infixExpression.getLeftOperand());

			// deal with right hand operand
			findLoopExpressionVariables(infixExpression.getRightOperand());

			// check if there is any extended operands and if so then handle
			// them also
			List list = infixExpression.extendedOperands();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				Expression expr = (Expression) iterator.next();
				// find variables in expression
				findLoopExpressionVariables(expr);
			}

		}

		// if this is a simple name then we got to a variable so we save it
		if (loopExp instanceof SimpleName) {
			SimpleName simpleName = (SimpleName) loopExp;
			// add the variable to the variables list
			_loopIterator.add(simpleName.getIdentifier());
		}
	}

	/**
	 * Runs over all statements inside the loop block and checks if any of the
	 * loop expression variables is being modified it keeps only the modified
	 * ones and delete all others.
	 *
	 * @param node the node
	 */
	private void findLoopIterator(Statement node) {

		// create a new variant variables detector
		BlockVariantVariablesDetector bVariablesDetector = new BlockVariantVariablesDetector();
		// run the detector on the block
		node.accept(bVariablesDetector);
		// get the list of variables from the detector
		ArrayList<String> variables = bVariablesDetector.get_variables();

		// Run over the loop iterators suggested list and check if they are
		// being updated inside the loop block if not then remove them from the
		// loop invariant variables
		_loopIterator.retainAll(variables);

	}

	/**
	 * Runs through the AST tree of the given block and collects all invariant
	 * variables and saves them to invariant variables list.
	 *
	 * @param node AST Block node
	 */
	private void findBlockInvariantVariables(Statement node) {

		// create a new invariant variables detector
		BlockInvariantVariablesDetector bInvariantVariablesDetector = new BlockInvariantVariablesDetector();
		// Run the detector on the block
		node.accept(bInvariantVariablesDetector);
		// save the invariant set
		_blockInvariantVariables = bInvariantVariablesDetector.getInvariantVariables();

	}

	/**
	 * Check if an expression is an invariant variable that is: a constant
	 * value, an invariant variable, invariant array access, invariant infix
	 * expression.
	 *
	 * @param exp the exp
	 * @return true, if is invariant
	 */
	@SuppressWarnings("unchecked")
	private boolean isInvariant(Expression exp) {
		// check if constant value
		if (isConstantValue(exp)) {
			return true;
		}
		// check if simple name
		if (exp instanceof SimpleName) {
			String name = ((SimpleName) exp).getIdentifier();
			if (_blockInvariantVariables.contains(name)) {
				return true;
			} else {
				return false;
			}
		}
		// check if array access and appears in invariant list
		if (exp instanceof ArrayAccess) {
			ArrayAccess access = (ArrayAccess) exp;
			if (_blockInvariantVariables.contains(access.toString())) {
				return true;
			} else {
				return false;
			}
		}
		// check if infix expression is invariant
		if (exp instanceof InfixExpression) {
			InfixExpression infixExpression = (InfixExpression) exp;

			// check if left operand and right operand are invariant
			if (isInvariant(infixExpression.getLeftOperand())
					&& isInvariant(infixExpression.getRightOperand())) {
				// rotate over all extended operands and check if they are
				// invariant
				List list = infixExpression.extendedOperands();
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Expression expr = (Expression) iterator.next();

					if (!isInvariant(expr)) {
						return false;
					}
				}
				return true;
			} else {
				return false;
			}
		}

		return false;
	}
}
