/**
 * @author Isaeed Mohanna
 * May 29, 2009 2009
 * 11:41:46 AM
 */
package jdtcomments;

import org.eclipse.jdt.core.dom.ArrayCreation;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.ClassInstanceCreation;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;

import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * This class is used to detect memory misuse which is situation in which memory
 * is allocated inside a loop Wither directly inside the loop body or inside a
 * function call. 1) detect a loop 2) detect array creation inside the loop 3)
 * check visit function calls inside the loop
 * 
 */
public class MemoryMisuseDetector extends BaseCommentDetector {

	// ============================= Private data members

	/** Indicates if we are inside a method. */
	boolean _isInsideMethodDeclaration;

	// ============================= Overridden functions
	
	/* Add a problem marker and highlight the text
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarker(int, int)
	 */
	/**
	 * Place problem marker specific.
	 *
	 * @param position position to place the marker
	 * @param length length of code to highlight
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) {
		// Place Marker
		getMarker().placeProblem(position, length, CommentTypes.MEMORY_MISUSE);
		
	}

	// ============================= Overridden Visitors

	// ============== Method declaration

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .MethodDeclaration)
	 */
	@Override
	public void endVisit(MethodDeclaration node) {
		// leaving method declaration
		_isInsideMethodDeclaration = false;
		super.endVisit(node);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * MethodDeclaration)
	 */
	@Override
	public boolean visit(MethodDeclaration node) {
		// Entering a method declaration
		_isInsideMethodDeclaration = true;
		return super.visit(node);
	}

	// ============== Variable declaration fragment

	/*
	 * 2 Cases; 1) Declaration of String variable - run through all initializers
	 * and check if they use string literals and method invocations only. 2) any
	 * class - check if it is initialized to a method invocation
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.VariableDeclarationFragment)
	 */
	@Override
	public boolean visit(VariableDeclarationFragment node) {
		// check if we are inside a method declaration
		if(_isInsideMethodDeclaration)
		{
			// check if the initializer allocates memory
			if(isMemoryAllocated(node.getInitializer()))
			{
				// found memory allocation inside a loop then place a marker
				placeProblemMarker(node.getParent().getStartPosition(),node.getParent().getLength());
			}
		}
		return super.visit(node);
	}
	
	// ============== Assignment
	
	/* We check if the assignment includes memory allocation
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.Assignment)
	 */
	@Override
	public boolean visit(Assignment node) {
		
		// check if we are inside a method declaration
		if(_isInsideMethodDeclaration)
		{
			
			// check if right side of the assignment allocates memory
			if(isMemoryAllocated(node.getRightHandSide()))
			{
				// found memory allocation inside a loop then place a marker
				placeProblemMarker(node.getStartPosition(),node.getLength());
			}
		}
		
		return super.visit(node);
	}

	// ================= Private methods

	/**
	 * This method decides if a new memory is being allocated.
	 *
	 * @param exp initializer expression
	 * @return boolean - indicates if the variable declaration Initializers
	 * allocates new memory TRUE : Memory is being allocated FALES:
	 * Memory is not being allocated
	 */
	private boolean isMemoryAllocated(Expression exp) {
		// handle Array Creation
		if (exp instanceof ArrayCreation) {
			return true;
		}

		// Handle Class instance creation
		if (exp instanceof ClassInstanceCreation) {
			return true;
		}

		// Handle method invocation
		if (exp instanceof MethodInvocation) {
			return true;
		}

		// handle infix expression
		if (exp instanceof InfixExpression) {
			return doesInfixExpressionAllocateMemory((InfixExpression) exp);
		}

		// Handle string literal
		if (exp instanceof StringLiteral) {
			return true;
		}

		// Any other case
		return false;
	}
	
	/**
	 * Checks if the infix expression allocates memory.
	 *
	 * @param exp the exp
	 * @return TRUE - allocates memory, FALSE otherwise
	 */
	private boolean doesInfixExpressionAllocateMemory(InfixExpression exp) {
		
		// if at least one operand allocates memory then memory is allocated
		if (isMemoryAllocated(exp.getLeftOperand()) || isMemoryAllocated(exp.getRightOperand())) {
			return true;
		}
		return false;
	}

}
