/*
 * 
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jdt.core.dom.ClassInstanceCreation;
import org.eclipse.jdt.core.dom.TypeDeclaration;

import utils.ObjectMarkerLocation;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * The Class ObjectInliningDetector.
 * 
 * @author Eyal Maderer
 * 
 *         <p>
 *         The Class ObjectInliningDetector.
 *         <p>
 *         <p>
 *         In our implementation of ObjectInliningDetector.java we extended
 *         our base comment detector to detect all the initiations of classes
 *         which are part of the selected project, we hold 2 collections, one
 *         for all the classes in the project that are relatively small (less
 *         than 4 methods and 4 fields), the other one is for all the places
 *         where a class instance was created, in the end we match the places
 *         where class instance was created with our collection of small
 *         classes, in the places where small class instance was created we
 *         supply a proper comment that in this place object inlining is
 *         recommended, instead of creating a new class instance and holding a
 *         pointer to it the user should take all the fields and methods of the
 *         small class and combine them to the creating class.
 *         <p>
 */
public class ObjectInliningDetector extends BaseCommentDetector 
{

	/** The m_col classes instances. */
	private static Collection<ObjectMarkerLocation> m_colClassesInstances = new ArrayList<ObjectMarkerLocation>();
	
	/** The m_col classes in program. */
	private static Collection<String> m_colClassesInProgram = new ArrayList<String>();
	
	/** The m_s current class name. */
	private String m_sCurrentClassName = null;
	
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarkerSpecific(int, int)
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) 
	{
		getMarker().placeProblem(position, length, CommentTypes.OBJECT_INLINING);
	}

	/**
	 * Place problem marker.
	 *
	 * @param position the position
	 * @param length the length
	 * @param marker the marker
	 * @param additionalCommant the additional commant
	 */
	protected void placeProblemMarker(int position, int length, MarkerCreator marker, String additionalCommant) 
	{
		if(isInsideCriticalFunction())
		{
			marker.placeProblem(position, length, CommentTypes.OBJECT_INLINING, additionalCommant);
		}
	}


	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.ClassInstanceCreation)
	 */
	@Override
	public boolean visit(ClassInstanceCreation node)
	{
		String sClassInstace = node.getType().toString();
		if(sClassInstace.endsWith(m_sCurrentClassName))
		{
			return false;
		}
		ObjectMarkerLocation classLoaction = new ObjectMarkerLocation(sClassInstace, node.getStartPosition(),node.getLength(), getMarker());
		boolean exists = false;
		for(ObjectMarkerLocation cLocation:m_colClassesInstances)
		{
			if(cLocation.equals(classLoaction))
			{
				exists = true;
				break;
			}
		}
		if(!exists)
		{
			ObjectMarkerLocation location = new ObjectMarkerLocation(sClassInstace, node.getStartPosition(),node.getLength(), getMarker());
			location.setClassDeclaredName(m_sCurrentClassName);
			m_colClassesInstances.add(location);
		}
		return true;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.ClassInstanceCreation)
	 */
	@Override
	public void endVisit(ClassInstanceCreation node)
	{
		String sClass = node.getType().toString();
		if(m_colClassesInProgram.contains(sClass) && !m_sCurrentClassName.equals(node.getType().toString()))
		{
			String comment = getComment(sClass, m_sCurrentClassName);
			placeProblemMarker(node.getStartPosition(),node.getLength(), getMarker(), comment);
		}
	}

	/**
	 * Gets the comment.
	 *
	 * @param sClass the s class
	 * @param declaredClass the declared class
	 * @return the comment
	 */
	private String getComment(String sClass, String declaredClass) 
	{
		String comment = "\n class " + sClass + " is a relatively small class" + 
		"\n you can combine class " + declaredClass + " and " + sClass +
		" to a new larger Class" + 
		"\n make the members and methods of class " + sClass + " Members of class " + 
		declaredClass + "\nAnd name the new class as " + declaredClass + "_" +  sClass;
		return comment;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.TypeDeclaration)
	 */
	@Override
	public boolean visit(TypeDeclaration node)
	{
		m_sCurrentClassName = node.getName().toString();
		if(!m_colClassesInProgram.contains(m_sCurrentClassName))
		{
			if(node.getMethods().length <= 4 && node.getFields().length <= 4)
			{
				m_colClassesInProgram.add(m_sCurrentClassName);
			}
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.TypeDeclaration)
	 */
	@Override
	public void endVisit(TypeDeclaration node)
	{
		for(ObjectMarkerLocation classLocation:m_colClassesInstances)
		{
			if(classLocation.getObjectName().equals(node.getName().toString()))
			{
				String comment = getComment(classLocation.getObjectName(), classLocation.getClassDeclaredName());
				placeProblemMarker(classLocation.getStartLocation(), classLocation.getLength(), classLocation.getMarker(), comment);
			}
		}
	}
	
	/**
	 * Reset.
	 */
	public static void reset()
	{
		m_colClassesInProgram.clear();
		m_colClassesInstances.clear();
	}

}
