/**
 * @author Isaeed Mohanna
 * Jul 17, 2009 2009
 * 1:21:16 AM
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jdt.core.dom.ArrayAccess;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.SynchronizedStatement;
import org.eclipse.jdt.core.dom.Modifier.ModifierKeyword;

import utils.BlockVariablesDetector;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * This class is used to detected cases in which and exclusive lock is used
 * while a shared lock could have been used in other words it identify all
 * Synchronized blocks with only read commands to global variables in them.
 */
public class SharedReadLockDetector extends BaseCommentDetector {

	// ============================= Private data members

	/**
	 * This methods tells the nested number of synchronized blocks at our given
	 * location. a counter is used instead of a regular boolean flag in order to
	 * detect nested loops
	 */
	int _syncCounter = 0;

	/** Holds a set of variables declared inside the synchronized block. */
	ArrayList<String> _blockVariables;

	/**
	 * This is used to store the result of our analysis. in case a global
	 * variable is modified inside the block the result is turned false
	 */
	boolean _isSharedRead = true;

	// ============================= Overridden functions
	
	/* Add a problem marker and highlight the text
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarker(int, int)
	 */
	/**
	 * Place problem marker specific.
	 *
	 * @param position position to place the marker
	 * @param length length of code to highlight
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) {
		// Place Marker
		getMarker().placeProblem(position, length, CommentTypes.USE_NON_EXCLUSIVE_LOCK);
		
	}

	// ============================= Overridden Visitors
	// ============== Synchronized statement
	/*
	 * Handles end of a synchronized statement
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public void endVisit(SynchronizedStatement node) {

		// Decrease the loop counter
		_syncCounter--;

		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// check the result
			if (_isSharedRead) {
				placeProblemMarker(node.getStartPosition(),12);
			}

			// Clear invariant variables list
			_blockVariables.clear();
			// Reset the result flag
			_isSharedRead = true;
		}
		super.endVisit(node);
	}

	/*
	 * Handles the begin of a synchronized statement
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public boolean visit(SynchronizedStatement node) {
		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// build invariant code set from block at first loop
			findBlockVariables(node);
		}

		// Increase the loop counter
		_syncCounter++;

		return super.visit(node);
	}

	// ============== Method declaration

	/*
	 * Leaves the method declaration class
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void endVisit(MethodDeclaration node) {
		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// Decrease the loop counter
				_syncCounter--;

				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// check the result
					if (_isSharedRead) {
						placeProblemMarker(modifier.getStartPosition(),12);
					}
					// Clear invariant variables list
					_blockVariables.clear();
					// Reset the result flag
					_isSharedRead = true;
				}
			}
		}

		super.endVisit(node);
	}

	/*
	 * Enters a method declaration so it checks if the method is declared
	 * synchronized and acts accordingly
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(MethodDeclaration node) {

		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// build invariant code set from block at first loop
					findBlockVariables(node.getBody());
				}

				// Increase the loop counter
				_syncCounter++;
			}
		}

		return super.visit(node);
	}

	// ============== Assignment
	/*
	 * Checks if left side of the assignment is a local variable
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment
	 * )
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.Assignment)
	 */
	@Override
	public boolean visit(Assignment node) {

		// if we are inside a synchronized block
		if (_syncCounter > 0) {

			// Left hand operand is a variable
			if (node.getLeftHandSide() instanceof SimpleName) {
				// cast
				SimpleName name = (SimpleName) node.getLeftHandSide();

				// check if left hand side is not a local variable then we are
				// writing to global variable
				if (!_blockVariables.contains(name.getIdentifier())) {
					// change the result status to false
					_isSharedRead = false;
				}
			}
			// Get left hand operand and check if it is an array access
			if (node.getLeftHandSide() instanceof ArrayAccess) {

				ArrayAccess arrayAccess = (ArrayAccess) node.getLeftHandSide();
				// get array access name
				String arrayName = getArrayName(arrayAccess);
				// check if left hand side is not a local variable then we are
				// writing to global variable
				if (!_blockVariables.contains(arrayName)) {
					// change the result status to false
					_isSharedRead = false;
				}

			}

		}
		return super.visit(node);
	}

	// ============================= Private Help methods

	/**
	 * Runs through the AST tree of the given block and collects all invariant
	 * variables and saves them to invariant variables list.
	 *
	 * @param node AST Block node
	 */
	private void findBlockVariables(Statement node) {

		// create a new invariant variables detector
		BlockVariablesDetector blockVariablesDetector = new BlockVariablesDetector();
		// Run the detector on the block
		node.accept(blockVariablesDetector);
		// save the invariant set
		_blockVariables = blockVariablesDetector.getVariables();

	}

	/**
	 * Returns the array name by taking the array access.
	 *
	 * @param access the access
	 * @return String the array name
	 */
	private String getArrayName(ArrayAccess access) {
		if (access.getArray() instanceof SimpleName) {
			return ((SimpleName) access.getArray()).getIdentifier();
		} else {
			if (access.getArray() instanceof ArrayAccess) {
				return getArrayName((ArrayAccess) access.getArray());
			}
		}
		return null;
	}

}
