/*
 * 
 */
package jdtcomments;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.PostfixExpression;
import org.eclipse.jdt.core.dom.PrefixExpression;
import org.eclipse.jdt.core.dom.SimpleName;

import utils.CommentTypes;
import utils.ObjectMarkerLocation;

// TODO: Auto-generated Javadoc
/**
 * * @author Eyal Maderer
 * 
 * <p>The Class StrengthReductionDetector.<p>
 * 
 *<p>In our implementation of AlgebraicSimplificationDetector.java we extended our base comment detector. 
 * The implementation of this comment is very easy, we just need to find all the Infix Expressions in the code 
 * and when we do, we need to check if one of the operands of this expression is zero, if so, we need to 
 * check the operators type and supply the suitable comment. <p>
 */
public class StrengthReductionDetector extends BaseCommentDetector 
{
	
	/** The _mp loop variables. */
	private Map<Integer,Collection<String>> _mpLoopVariables = new HashMap<Integer,Collection<String>> ();

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarkerSpecific(int, int)
	 */
	protected void placeProblemMarkerSpecific(int position, int length) 
	{
		getMarker().placeProblem(position, length, CommentTypes.STRENGTH_REDUCTION);
	}

	/**
	 * Place problem marker.
	 *
	 * @param position the position
	 * @param length the length
	 * @param marker the marker
	 * @param additionalComment the additional comment
	 */
	protected void placeProblemMarker(int position, int length, MarkerCreator marker, String additionalComment) 
	{
		if(isInsideCriticalFunction())
		{
			marker.placeProblem(position, length, CommentTypes.STRENGTH_REDUCTION, additionalComment);
		}
	}

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.PostfixExpression)
	 */
	@Override
	public boolean visit(PostfixExpression node)
	{
		super.visit(node);
		if(node.getParent() instanceof ForStatement)
		{
			Collection<String> colVariables = null;
			if(null == getLoopVars().get(getLoopCount()))
			{
				colVariables = new ArrayList<String>();
			}
			else
			{
				colVariables = getLoopVars().get(getLoopCount());
			}
			colVariables.add(node.getOperand().toString());
			getLoopVars().put(getLoopCount(), colVariables);
		}
		return true;
	}
	
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.PrefixExpression)
	 */
	@Override
	public boolean visit(PrefixExpression node)
	{
		super.visit(node);
		if(node.getParent() instanceof ForStatement)
		{
			Collection<String> colVariables = null;
			if(null == getLoopVars().get(getLoopCount()))
			{
				colVariables = new ArrayList<String>();
			}
			else
			{
				colVariables = getLoopVars().get(getLoopCount());
			}
			colVariables.add(node.getOperand().toString());
			getLoopVars().put(getLoopCount(), colVariables);
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.InfixExpression)
	 */
	@Override
	public boolean visit(InfixExpression node)
	{
		try
		{
			checkForMultiplyOrDevidion(node);
			checkForLoopVariables(node);
		}
		catch (Exception e) {
		}
		super.visit(node);
		return true;
	}

	/**
	 * Check for loop variables.
	 *
	 * @param node the node
	 */
	private void checkForLoopVariables(InfixExpression node)
	{
		if(!isInsideLoop())
		{
			return ;
		}
		if(!node.getOperator().toString().equalsIgnoreCase("*"))
		{
			return ;
		}
		for(Integer i:getLoopVars().keySet())
		{
			for(String var:getLoopVars().get(i))
			{
				if(var.toString().equalsIgnoreCase(node.getLeftOperand().toString()))
				{
					String message = "you can replace this with the declaration of \n " +
							"'newVariable = newVariable + " + node.getRightOperand() +";' inside the loop \n"
							+ "out side the loop with the declaration 'newVariable = 0;' \n"
							+ "and here you write newVariable instead " + node.toString();
					getObjectsLocations().add(new ObjectMarkerLocation(var, node.getStartPosition(), node.getLength(), getMarker(), message));			
				}
				if(var.toString().equalsIgnoreCase(node.getRightOperand().toString()))
				{
					String message = "you can replace this with the declaration of \n " +
							"'newVariable = newVariable + " + node.getLeftOperand() +";' inside the loop \n"
							+ "out side the loop with the declaration 'newVariable = 0;' \n"
							+ "and here you write newVariable instead " + node.toString();
					getObjectsLocations().add(new ObjectMarkerLocation(var, node.getStartPosition(), node.getLength(), getMarker(), message));			
				}
				
			}
		}
	}

	/**
	 * Check for multiply or devidion.
	 *
	 * @param node the node
	 */
	private void checkForMultiplyOrDevidion(InfixExpression node) 
	{
		String operator = node.getOperator().toString();
		if(!(operator.equalsIgnoreCase("*")||operator.equalsIgnoreCase("/")))
		{
			return;
		}
		Double num = null;
		String variable = null;
		if(node.getLeftOperand() instanceof SimpleName)
		{
			variable = node.getLeftOperand().toString();
		}
		
		if(node.getRightOperand() instanceof SimpleName)
		{
			variable = node.getRightOperand().toString();
		}
		
		if(node.getLeftOperand() instanceof NumberLiteral)
		{
			num = Double.valueOf(node.getLeftOperand().toString());
		}
		
		if(node.getRightOperand() instanceof NumberLiteral)
		{
			num = Double.valueOf(node.getRightOperand().toString());
		}
		
		if(null == num || null == variable)
		{
			return;
		}
		int x = findNumber(num);
		if(0 == x)
		{
			return;
		}
		operator = null;
		if(node.getOperator().toString().equalsIgnoreCase("*"))
		{
			operator = "<<";
		}
		else
		{
			operator = ">>";
		}
		String message = "You can replace this with \n" + variable + operator + x;
		placeProblemMarker(node.getStartPosition(), node.getLength(), getMarker(), message);
	}

	/**
	 * Find number.
	 *
	 * @param num the num
	 * @return the int
	 */
	private int findNumber(Double num) 
	{
		int x = 0;
		while((0 == num%2) && (num!=0))
		{
			x++;
			num/=2;
		}
		if(num != 1d)
		{
			return 0;
		}
		return x;
	}

	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.ForStatement)
	 */
	@Override
	public void endVisit(ForStatement node)
	{
		getLoopVariables().remove(getLoopCount());
		for(ObjectMarkerLocation mark:getObjectsLocations())
		{
			if(null != mark.getMessage() && !mark.isMarked())
			{
				mark.setMarked(true);
				placeProblemMarker(mark.getStartLocation(), mark.getLength(), mark.getMarker(), mark.getMessage());
			}
		}
		super.endVisit(node);

	}

	/**
	 * Sets the loop variables.
	 *
	 * @param _mpLoopVariables the _mp loop variables
	 */
	public void setLoopVariables(Map<Integer,Collection<String>> _mpLoopVariables)
	{
		this._mpLoopVariables = _mpLoopVariables;
	}

	/**
	 * Gets the loop vars.
	 *
	 * @return the loop vars
	 */
	public Map<Integer,Collection<String>> getLoopVars()
	{
		return _mpLoopVariables;
	}

	
}


