/**
 * @author Isaeed Mohanna
 * Jul 11, 2009 2009
 * 3:15:00 AM
 */
package jdtcomments;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.dom.ArrayAccess;
import org.eclipse.jdt.core.dom.ArrayCreation;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.BooleanLiteral;
import org.eclipse.jdt.core.dom.CharacterLiteral;
import org.eclipse.jdt.core.dom.ClassInstanceCreation;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.NullLiteral;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.SynchronizedStatement;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jdt.core.dom.Modifier.ModifierKeyword;

import utils.BlockInvariantVariablesDetector;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * This class find invariant code inside a synchronized block code it works the
 * same way as the invariant code detector for loops.
 */
public class SynchronizedBlockInvariantCodeDetector extends BaseCommentDetector {

	// ============================= Private data members

	/**
	 * This methods tells the nested number of synchronized blocks at our given
	 * location. a counter is used instead of a regular boolean flag in order to
	 * detect nested loops
	 */
	int _syncCounter = 0;

	/** Holds a set of invariant variables inside the current block, it is filled upon first entrance to a synchronized block. */
	Set<String> _blockInvariantVariables;

	// ============================= Overridden functions
	
	/* Add a problem marker and highlight the text
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarker(int, int)
	 */
	/**
	 * Place problem marker specific.
	 *
	 * @param position position to place the marker
	 * @param length length of code to highlight
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) {
		// Place Marker
		getMarker().placeProblem(position, length, CommentTypes.INVARIANT_CODE_WITH_SYNCHRONIZED);
		
	}

	// ============================= Overridden Visitors
	// ============== Synchronized statement

	/*
	 * Handles end of a synchronized statement
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public void endVisit(SynchronizedStatement node) {

		// Decrease the loop counter
		_syncCounter--;

		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// Clear invariant variables list
			_blockInvariantVariables.clear();
		}
		super.endVisit(node);
	}

	/*
	 * Handles the begin of a synchronized statement
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public boolean visit(SynchronizedStatement node) {
		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// build invariant code set from block at first loop
			findBlockInvariantVariables(node);
		}

		// Increase the loop counter
		_syncCounter++;

		return super.visit(node);
	}

	// ============== Method Invocation

	/*
	 * if the method invocation is inside the code then this is a suspected
	 * place for invariant code
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * MethodInvocation)
	 */

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.MethodInvocation)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(MethodInvocation node) {

		// check that we are inside a loop
		if (_syncCounter > 0) {
			boolean result = true;
			// check if the method arguments are invariant
			List arguments = node.arguments();
			for (Iterator iterator = arguments.iterator(); iterator.hasNext();) {
				Expression exp = (Expression) iterator.next();
				if (!isInvariantVariable(exp)) {
					result = false;
				}
			}
			// if all arguments list is invariant then we will warn that this
			// call might be invariant and the user should check it manually
			if (result) {
				placeMarker(node.getStartPosition(),node.getLength(), true);
			}
		}

		return super.visit(node);
	}

	// ============== Class Instance Creation

	/*
	 * Class creation is another suspect place for invariant code - we will add
	 * a marker and advice the programmer to check if the creation is invariant
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * ClassInstanceCreation)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.ClassInstanceCreation)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(ClassInstanceCreation node) {
		// check that we are inside a loop
		if (_syncCounter > 0) {
			boolean result = true;
			// check if the method arguments are invariant
			List arguments = node.arguments();
			for (Iterator iterator = arguments.iterator(); iterator.hasNext();) {
				Expression exp = (Expression) iterator.next();
				if (!isInvariantVariable(exp)) {
					result = false;
				}
			}
			// if all arguments list is invariant then we will warn that this
			// call might be invariant and the user should check it manually
			if (result) {
				placeMarker(node.getStartPosition(),node.getLength(), true);
			}
		}

		return super.visit(node);
	}

	// ============== Method declaration

	/*
	 * Leaves the method declaration class
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void endVisit(MethodDeclaration node) {
		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// Decrease the loop counter
				_syncCounter--;

				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// Clear invariant variables list
					_blockInvariantVariables.clear();
				}
			}
		}

		super.endVisit(node);
	}

	/*
	 * Enters a method declaration so it checks if the method is declared
	 * synchronized and acts accordingly
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(MethodDeclaration node) {

		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// build invariant code set from block at first loop
					findBlockInvariantVariables(node.getBody());
				}

				// Increase the loop counter
				_syncCounter++;
			}
		}

		return super.visit(node);
	}

	// ============== Variable Declaration Fragment
	/*
	 * Checks if there is a variable declartion fragment inside a loop and is
	 * Initialized to a constant value(non-Javadoc)
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * VariableDeclarationFragment)
	 */
	@Override
	public boolean visit(VariableDeclarationFragment node) {
		// check if we are inside a loop
		if (_syncCounter > 0) {
			// check if initializer is an invariable expression
			if (isInvariantVariable(node.getInitializer())) {
				// the value has been initialized to an invariant statement
				// so we check if the variable is invariant
				if (_blockInvariantVariables.contains(node.getName().getIdentifier())) {
					// if the variable is invariant then print a warning message
					placeProblemMarker(node.getParent().getStartPosition(),node.getParent().getLength());
				}
			}
		}
		return super.visit(node);
	}

	// ============== Assignment
	/*
	 * Visits the assignment nodes (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment
	 * )
	 */
	@Override
	public boolean visit(Assignment node) {
		// check if we are inside a loop
		if (_syncCounter > 0) {
			// Check Left side of the is an invariant variable
			if (isInvariantVariable(node.getLeftHandSide())) {
				// this is an invariant variable if the right side consists of
				// constants and invariant variables
				if (isInvariantVariable(node.getRightHandSide())) {
					// invariant variable assigned to an invariant expression
					placeProblemMarker(node.getStartPosition(),node.getLength());

				} else {
					// invariant assigned a variant code. so we do not add a
					// warning

				}

			} else {
				// The left side variable is not invariant then the statement is
				// not invariant
			}
		}
		return super.visit(node);
	}

	// ============================= Private Help methods

	/**
	 * This method checks if the variable is invariant according to the
	 * following conditions: 1. is defined outside the loop 2. appears in the
	 * invariantVariables map
	 *
	 * @param exp the exp
	 * @return TRUE - Invariant variable FALSE - non invariant variable
	 */
	@SuppressWarnings("unchecked")
	private boolean isInvariantVariable(Expression exp) {
		// check if expression is a constant value
		if (isConstantValue(exp)) {
			return true;
		}

		// check if this is an infix expression
		if (exp instanceof InfixExpression) {
			InfixExpression infixExpression = (InfixExpression) exp;

			// check if left side operand is invariant
			if (isInvariantVariable(infixExpression.getLeftOperand())
					&& isInvariantVariable(infixExpression.getRightOperand())) {
				// if left and right side of the infix expression are invariant
				// then we check the extended operands
				List list = infixExpression.extendedOperands();
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Expression expr = (Expression) iterator.next();

					// check if the expression is an invariant code
					if (!isInvariantVariable(expr)) {
						return false;
					}
				}
				return true;

			} else {
				return false;
			}
		}

		// check type of expression
		if (exp instanceof SimpleName) {
			SimpleName name = (SimpleName) exp;

			// check if variable appears in invariant variables map
			if (_blockInvariantVariables.contains(name.getIdentifier())) {
				// variable appears in invariants list
				return true;

			} else {
				return false;
			}
		}
		// check for array access
		if (exp instanceof ArrayAccess) {

			ArrayAccess access = (ArrayAccess) exp;

			// check if array name appears inside the invariant variables list
			String arrayName = access.toString();
			if (_blockInvariantVariables.contains(arrayName)) {
				return true;
			} else {
				return false;
			}
		}

		// check if array creation
		if (exp instanceof ArrayCreation) {
			// check if the array name appears inside the invariant variables
			// list
			if (exp.getParent() instanceof VariableDeclarationFragment) {
				VariableDeclarationFragment vdFragment = (VariableDeclarationFragment) exp
						.getParent();
				String arrayName = vdFragment.getName().getIdentifier();

				// check if name in invariant list
				if (_blockInvariantVariables.contains(arrayName)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check if the right side of the assignment is a constant value that is
	 * some literal type.
	 *
	 * @param exp Expression
	 * @return TRUE - A constant Value FALSE - Variable or expression
	 */
	private boolean isConstantValue(Expression exp) {
		// check for null
		if (exp == null) {
			return false;
		}
		// check for different types of literals
		if (exp instanceof NumberLiteral || exp instanceof StringLiteral
				|| exp instanceof BooleanLiteral || exp instanceof NullLiteral
				|| exp instanceof CharacterLiteral) {
			// if it is a literal type
			return true;
		}

		return false;
	}

	/**
	 * Runs through the AST tree of the given block and collects all invariant
	 * variables and saves them to invariant variables list.
	 *
	 * @param node AST Block node
	 */
	private void findBlockInvariantVariables(Statement node) {

		// create a new invariant variables detector
		BlockInvariantVariablesDetector bInvariantVariablesDetector = new BlockInvariantVariablesDetector();
		// Run the detector on the block
		node.accept(bInvariantVariablesDetector);
		// save the invariant set
		_blockInvariantVariables = bInvariantVariablesDetector.getInvariantVariables();

	}

	/**
	 * This method is used to add the warning message for a possible invariant
	 * method invocation.
	 *
	 * @param start the start
	 * @param length the length
	 * @param possible the possible
	 */
	private void placeMarker(int start,int length, boolean possible) {

		// place initiating array inside loops error message
		if(isInsideCriticalFunction())
		{
			getMarker().placeProblem(start, length, utils.CommentTypes.POSSIBLE_INVARIANTCODE);
		}

	}

}
