/**
 * @author Isaeed Mohanna
 * Jul 16, 2009 2009
 * 11:22:45 PM
 */
package jdtcomments;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.dom.ArrayAccess;
import org.eclipse.jdt.core.dom.ArrayCreation;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.BooleanLiteral;
import org.eclipse.jdt.core.dom.CharacterLiteral;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.NullLiteral;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.SynchronizedStatement;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jdt.core.dom.Modifier.ModifierKeyword;

import utils.BlockInvariantVariablesDetector;
import utils.CommentTypes;

// TODO: Auto-generated Javadoc
/**
 * This class searches for synchronized blocks and checks and in case of an
 * array or struck access it checks if it access only one cell of the array then
 * we need to reduce the lock granularity and lock the cell only.
 */
public class SynchronizedLockGranularityDetector extends BaseCommentDetector {

	// ============================= Private data members

	/**
	 * This methods tells the nested number of synchronized blocks at our given
	 * location. a counter is used instead of a regular boolean flag in order to
	 * detect nested loops
	 */
	int _syncCounter = 0;

	/** Holds a set of invariant variables inside the current block, it is filled upon first entrance to a synchronized block. */
	Set<String> _blockInvariantVariables;

	// ============================= Overridden functions
	
	/* Add a problem marker and highlight the text
	 * @see jdtcomments.BaseCommentDetector#placeProblemMarker(int, int)
	 */
	/**
	 * Place problem marker specific.
	 *
	 * @param position position to place the marker
	 * @param length length of code to highlight
	 */
	@Override
	protected void placeProblemMarkerSpecific(int position, int length) {
		// Place Marker
		getMarker().placeProblem(position, length, CommentTypes.REDUCE_LOCK_GRANULARITY);
		
	}

	// ============================= Overridden Visitors
	// ============== Synchronized statement
	/*
	 * Handles end of a synchronized statement
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public void endVisit(SynchronizedStatement node) {

		// Decrease the loop counter
		_syncCounter--;

		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// Clear invariant variables list
			_blockInvariantVariables.clear();
		}
		super.endVisit(node);
	}

	/*
	 * Handles the begin of a synchronized statement
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * SynchronizedStatement)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.SynchronizedStatement)
	 */
	@Override
	public boolean visit(SynchronizedStatement node) {
		// clear the variables map if this is a first level loop
		if (_syncCounter == 0) {
			// build invariant code set from block at first loop
			findBlockInvariantVariables(node);
		}

		// Increase the loop counter
		_syncCounter++;

		return super.visit(node);
	}

	// ============== Method declaration

	/*
	 * Leaves the method declaration class
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#endVisit(org.eclipse.jdt.core.dom
	 * .MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#endVisit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void endVisit(MethodDeclaration node) {
		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// Decrease the loop counter
				_syncCounter--;

				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// Clear invariant variables list
					_blockInvariantVariables.clear();
				}
			}
		}

		super.endVisit(node);
	}

	/*
	 * Enters a method declaration so it checks if the method is declared
	 * synchronized and acts accordingly
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * MethodDeclaration)
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.MethodDeclaration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(MethodDeclaration node) {

		// get Modifiers list
		List modifiers = node.modifiers();
		// check if modifier is synchronized
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			Modifier modifier = (Modifier) iterator.next();

			// check if modifier is synchronized
			if (modifier.getKeyword().equals(ModifierKeyword.SYNCHRONIZED_KEYWORD)) {
				// clear the variables map if this is a first level loop
				if (_syncCounter == 0) {
					// build invariant code set from block at first loop
					findBlockInvariantVariables(node.getBody());
				}

				// Increase the loop counter
				_syncCounter++;
			}
		}

		return super.visit(node);
	}

	// ============== Assignment
	/*
	 * This method checks if the left side of assignment is an array access or a
	 * struct and then checks if the array cell index is invariant and if so
	 * then we are accessing only one cell of the array so we can reduce the
	 * lock granularity
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment
	 * )
	 */
	/* (non-Javadoc)
	 * @see jdtcomments.BaseCommentDetector#visit(org.eclipse.jdt.core.dom.Assignment)
	 */
	@Override
	public boolean visit(Assignment node) {
		// only if we are inside a synchronized block
		if (_syncCounter > 0) {

			// Get left hand operand and check if it is an array access
			if (node.getLeftHandSide() instanceof ArrayAccess) {

				ArrayAccess arrayAccess = (ArrayAccess) node.getLeftHandSide();

				// check if array access index is invariant
				if (isArrayIndexesInvariant(arrayAccess)) {
					// we are accessing one cell of the array inside the block
					placeProblemMarker(arrayAccess.getStartPosition(),arrayAccess.getLength());
				}
			}
		}
		return super.visit(node);
	}

	// ============================= Private Help methods

	/**
	 * Runs through the AST tree of the given block and collects all invariant
	 * variables and saves them to invariant variables list.
	 *
	 * @param node AST Block node
	 */
	private void findBlockInvariantVariables(Statement node) {

		// create a new invariant variables detector
		BlockInvariantVariablesDetector bInvariantVariablesDetector = new BlockInvariantVariablesDetector();
		// Run the detector on the block
		node.accept(bInvariantVariablesDetector);
		// save the invariant set
		_blockInvariantVariables = bInvariantVariablesDetector.getInvariantVariables();

	}

	/**
	 * This method runs through array access indexes and returns true if all of
	 * them are invariant.
	 *
	 * @param arrayAccess the array access
	 * @return true, if is array indexes invariant
	 */
	private boolean isArrayIndexesInvariant(ArrayAccess arrayAccess) {
		// check if index is invariant
		if (isInvariantVariable(arrayAccess.getIndex())) {
			// check if there is another array
			if (arrayAccess.getArray() instanceof ArrayAccess) {
				return isArrayIndexesInvariant((ArrayAccess) arrayAccess.getArray());
			} else {
				return true;
			}

		} else {
			return false;
		}
	}

	/**
	 * This method checks if the variable is invariant according to the
	 * following conditions: 1. is defined outside the loop 2. appears in the
	 * invariantVariables map
	 *
	 * @param exp the exp
	 * @return TRUE - Invariant variable FALSE - non invariant variable
	 */
	@SuppressWarnings("unchecked")
	private boolean isInvariantVariable(Expression exp) {
		// check if expression is a constant value
		if (isConstantValue(exp)) {
			return true;
		}

		// check if this is an infix expression
		if (exp instanceof InfixExpression) {
			InfixExpression infixExpression = (InfixExpression) exp;

			// check if left side operand is invariant
			if (isInvariantVariable(infixExpression.getLeftOperand())
					&& isInvariantVariable(infixExpression.getRightOperand())) {
				// if left and right side of the infix expression are invariant
				// then we check the extended operands
				List list = infixExpression.extendedOperands();
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Expression expr = (Expression) iterator.next();

					// check if the expression is an invariant code
					if (!isInvariantVariable(expr)) {
						return false;
					}
				}
				return true;

			} else {
				return false;
			}
		}

		// check type of expression
		if (exp instanceof SimpleName) {
			SimpleName name = (SimpleName) exp;

			// check if variable appears in invariant variables map
			if (_blockInvariantVariables.contains(name.getIdentifier())) {
				// variable appears in invariants list
				return true;

			} else {
				return false;
			}
		}
		// check for array access
		if (exp instanceof ArrayAccess) {

			ArrayAccess access = (ArrayAccess) exp;

			// check if array name appears inside the invariant variables list
			String arrayName = access.toString();
			if (_blockInvariantVariables.contains(arrayName)) {
				return true;
			} else {
				return false;
			}
		}

		// check if array creation
		if (exp instanceof ArrayCreation) {
			// check if the array name appears inside the invariant variables
			// list
			if (exp.getParent() instanceof VariableDeclarationFragment) {
				VariableDeclarationFragment vdFragment = (VariableDeclarationFragment) exp
						.getParent();
				String arrayName = vdFragment.getName().getIdentifier();

				// check if name in invariant list
				if (_blockInvariantVariables.contains(arrayName)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check if the right side of the assignment is a constant value that is
	 * some literal type.
	 *
	 * @param exp Expression
	 * @return TRUE - A constant Value FALSE - Variable or expression
	 */
	private boolean isConstantValue(Expression exp) {
		// check for null
		if (exp == null) {
			return false;
		}
		// check for different types of literals
		if (exp instanceof NumberLiteral || exp instanceof StringLiteral
				|| exp instanceof BooleanLiteral || exp instanceof NullLiteral
				|| exp instanceof CharacterLiteral) {
			// if it is a literal type
			return true;
		}

		return false;
	}

}
