/**
 * @author Isaeed Mohanna
 * Jun 26, 2009 2009
 * 12:33:56 PM
 */
package utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.ArrayAccess;
import org.eclipse.jdt.core.dom.ArrayCreation;
import org.eclipse.jdt.core.dom.Assignment;
import org.eclipse.jdt.core.dom.BooleanLiteral;
import org.eclipse.jdt.core.dom.CharacterLiteral;
import org.eclipse.jdt.core.dom.ClassInstanceCreation;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.NullLiteral;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.PostfixExpression;
import org.eclipse.jdt.core.dom.PrefixExpression;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;

// TODO: Auto-generated Javadoc
/**
 * This class detects all invariant variables inside a block and saves them into
 * an array list that could be retrieved by the get invariant code command.
 */
public class BlockInvariantVariablesDetector extends ASTVisitor {

	// ===================== Private data members

	/**
	 * This methods tells the nested number of loops at our given location. a
	 * counter is used instead of a regular boolean flag in order to detect
	 * nested loops
	 */
	int _loopCounter = 0;

	/** Holds a list of all variables declared or referenced inside the block. */
	ArrayList<String> _variables = new ArrayList<String>();

	/**
	 * Holds a map of dependencies, where for each variable (key) there is an
	 * array list of all variables that affect the key variable. once we decide
	 * that a variable is invariant we remove it from the list
	 */
	HashMap<String, ArrayList<String>> _invariantVariables = new HashMap<String, ArrayList<String>>();

	// ===================== Getters
	/**
	 * Gets the invariant variables.
	 *
	 * @return Set the set of block invariant variables
	 */
	public Set<String> getInvariantVariables() {
		return _invariantVariables.keySet();
	}

	// ===================== Overridden Visitors
	// =========== Variable declaration

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.VariableDeclarationFragment)
	 */
	@SuppressWarnings("unchecked")
	/*
	 * Here we check if a declared variable is invariant by: 1) it is
	 * initialized to a constant value 2) it is initialized to an invariant
	 * variable
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * VariableDeclarationFragment)
	 */
	@Override
	public boolean visit(VariableDeclarationFragment node) {
		// get declared variable name
		String declaredVariable = node.getName().getIdentifier();
		// check if the initializer is a constant value
		if (isConstantValue(node.getInitializer())) {
			// if initializer is a constant value so for now it is invariant
			// (unless it is touched further ahead which will be identified
			// then)
			addInvariantVariableDependency(declaredVariable, null);
		}

		// check if the initializer is a simple variable
		if (node.getInitializer() instanceof SimpleName) {
			SimpleName name = (SimpleName) node.getInitializer();
			// check if variable is invariant
			if (_invariantVariables.containsKey(name.getIdentifier())) {
				// if variable is invariant then our declared variable is also
				// invariant so we add it to invariant list
				addInvariantVariableDependency(declaredVariable, name.getIdentifier());
			} else {
				// if variable is not invariant then we check if it appears in
				// variables list
				if (_variables.contains(name.getIdentifier())) {
					// It is a variant variable then so is our variable
				} else {
					// this is the first access to it then we assume it is
					// invariant and add it to the invariant list
					// add variable to invariant list
					addInvariantVariableDependency(name.getIdentifier(), null);
					// add the declared invariant to the lists
					addInvariantVariableDependency(declaredVariable, name.getIdentifier());
				}
			}
		}

		// check if the initializer is an infix expression
		if (node.getInitializer() instanceof InfixExpression) {
			InfixExpression iExpression = (InfixExpression) node.getInitializer();
			// saves infix expression variables list
			ArrayList<String> infVariables = new ArrayList<String>();
			// check if left operand and right operand is invariant or constant
			if (isInvariantVariable(iExpression.getLeftOperand())
					&& isInvariantVariable(iExpression.getRightOperand())) {
				// check if for all extended operations are invariant expression
				boolean result = true;
				// add the left and right variables to the list if needed
				addVariableNameToList(infVariables, iExpression.getLeftOperand());
				addVariableNameToList(infVariables, iExpression.getRightOperand());

				List list = iExpression.extendedOperands();
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Expression expr = (Expression) iterator.next();
					// if this is a variable then add it to variables list so it
					// can be added as a dependent
					addVariableNameToList(infVariables, expr);

					if (!isInvariantVariable(expr)) {
						result = false;
					}
				}

				if (result) {
					// this is an invariant expression then we add the variable
					// to the invariant list
					addInvariantVariableDependencyList(declaredVariable, infVariables);
				}
			}
		}

		// check if this is an array creation
		if (node.getInitializer() instanceof ArrayCreation) {
			// we check if the array declaration statement is move able outside
			// the loop block by checking if all dimensions are invariant
			boolean result = true;
			ArrayCreation arCreation = (ArrayCreation) node.getInitializer();
			// get dimensions list
			List dimesions = arCreation.dimensions();
			// saves array creation dependent values
			ArrayList<String> infVariables = new ArrayList<String>();
			// rotate over all dimensions and check if they are Invariant
			for (Iterator iterator = dimesions.iterator(); iterator.hasNext();) {
				Expression exp = (Expression) iterator.next();
				// if this is a variable then add it to variables list so it
				// can be added as a dependent
				if (exp instanceof SimpleName) {
					infVariables.add(((SimpleName) exp).getIdentifier());
				}

				if (!isInvariantVariable(exp)) {
					result = false;
				}
			}
			if (result) {
				// this is an invariant expression then we add the variable
				// to the invariant list
				addInvariantVariableDependencyList(declaredVariable, infVariables);
			} else {
				// it is variant code so we add it only to the variables list
				_variables.add(declaredVariable);
			}
		}

		// Handle the case of array access initializer
		if (node.getInitializer() instanceof ArrayAccess) {
			// check first if the array instance as a whole is variant
			ArrayAccess access = (ArrayAccess) node.getInitializer();
			String arrayName = getArrayName(access);
			// check first if the array instance itself is not invariant
			if (_invariantVariables.containsKey(arrayName) || !_variables.contains(arrayName)) {
				// check if array access is invariant
				if (_invariantVariables.containsKey(access.toString())) {
					// array access is invariant then we add the variable to the
					// invariant list with the array access as a dependent
					addInvariantVariableDependency(declaredVariable, access.toString());
				} else {
					// array access is not in invariant list so we check if it
					// is in
					// variables list which will mean that it is variant
					if (_variables.contains(access.toString())) {
						// this means that the declared variable is variant
					} else {
						// this is a new array access so we check if the array
						// index
						// is invariant
						if (isInvariantVariable(access)) {
							// Add the declared variable with the array access
							// as a
							// dependent
							addInvariantVariableDependency(declaredVariable, access.toString());
							// get the variable index list of the array access
							ArrayList<String> aaVar = new ArrayList<String>();
							getArrayAccessIndexVariablesList(aaVar, access);
							// add the array access to invariant list with its
							// indexes as dependents
							addInvariantVariableDependencyList(access.toString(), aaVar);

						}
					}
				}
			}
		}

		// check if the initializer is a class instance creation
		if (node.getInitializer() instanceof ClassInstanceCreation) {
			// TODO: should i handle class creation and calls!!!!
		}

		return super.visit(node);
	}

	// =========== PostfixExpression
	/*
	 * Handle the case in which we are changing the variable value using a post
	 * fix expression e.g. var++, var--
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * PostfixExpression)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.PostfixExpression)
	 */
	@Override
	public boolean visit(PostfixExpression node) {

		// check for a simple name
		if (node.getOperand() instanceof SimpleName) {
			SimpleName name = (SimpleName) node.getOperand();
			// remove the variable from the invariant list
			removeInvariantVariable(name.getIdentifier());
		}

		return super.visit(node);
	}

	// =========== PrefixExpression
	/*
	 * Handle the case in which we are changing the variable value using a
	 * prefix expression e.g. ++var, --var
	 * 
	 * @seeorg.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.
	 * PrefixExpression)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.PrefixExpression)
	 */
	@Override
	public boolean visit(PrefixExpression node) {
		// check for a simple name
		if (node.getOperand() instanceof SimpleName) {
			SimpleName name = (SimpleName) node.getOperand();
			// remove the variable from the invariant list
			removeInvariantVariable(name.getIdentifier());
		}
		return super.visit(node);
	}

	// =========== Assignment

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jdt.core.dom.ASTVisitor#visit(org.eclipse.jdt.core.dom.Assignment
	 * )
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean visit(Assignment node) {

		// Check the right hand side of the assignment
		Expression rightExp = node.getRightHandSide();

		// check if right expression is a constant value
		if (isConstantValue(rightExp)) {
			// since right hand expression is a constant value then we check the
			// left side of the expression
			Expression leftExp = node.getLeftHandSide();
			if (leftExp instanceof SimpleName) {
				SimpleName leftName = (SimpleName) leftExp;
				// check if name already appear in invariant list
				if (_invariantVariables.containsKey(leftName.getIdentifier())) {
					// if so then there is nothing to change
				} else {
					// variable is not in invariant list then we check if it
					// appears in variable list
					if (_variables.contains(leftName.getIdentifier())) {
						// Variable appears in variables list but doesn't appear
						// in invariant list then it is variant
					} else {
						// variable doesn't appear in both variable and
						// invariant
						// list then this is its first appearance in the block
						// and since it is defined outside the block and
						// assigned a constant value then it is invariant
						// variable
						_variables.add(leftName.getIdentifier());
						addInvariantVariableDependency(leftName.getIdentifier(), null);
					}
				}
			}
			// check if left side is an array access
			if (leftExp instanceof ArrayAccess) {
				ArrayAccess access = (ArrayAccess) leftExp;
				// check if array access is invariant
				if (isInvariantVariable(access)) {
					// find array access variables
					ArrayList<String> aaVar = new ArrayList<String>();
					getArrayAccessIndexVariablesList(aaVar, access);
					// add it to invariant list if it is not there
					addInvariantVariableDependencyList(access.toString(), aaVar);

				} else {
					// it is invariant so add it only to variables list
					_variables.add(access.toString());
				}
			}
		}

		// check if right hand side is a simple name
		if (rightExp instanceof SimpleName) {
			SimpleName rightName = (SimpleName) rightExp;
			// check if variable has been read before
			if (!_variables.contains(rightName.getIdentifier())) {

				// add the variable to the variables list
				_variables.add(rightName.getIdentifier());
				// we assume that it is invariant variable unless proved
				// otherwise since we are in right hand side then the
				// variable is not related any other variable
				addInvariantVariableDependency(rightName.getIdentifier(), null);

			}
			// get left hand side of the assignment
			Expression leftExp = node.getLeftHandSide();
			// check if left side is a variable
			if (leftExp instanceof SimpleName) {
				SimpleName leftName = (SimpleName) leftExp;
				// check if variable is invariant
				if (_invariantVariables.containsKey(leftName.getIdentifier())) {
					// if the variant is already an invariant then we check if
					// right side is invariant
					if (_invariantVariables.containsKey(rightName.getIdentifier())) {
						// invariant variable assigned to an invariant variable
						// we only need to add the dependency
						addInvariantVariableDependency(leftName.getIdentifier(), rightName
								.getIdentifier());
					} else {
						// if left side is invariant but right side is not
						// invariant then left side is not invariant anymore and
						// needs to be removed from invariant list
						removeInvariantVariable(leftName.getIdentifier());
					}
				} else {
					// left side of the assignment is not invariant
					// check if it is in variable list i.e. it is not invariant
					if (_variables.contains(leftName.getIdentifier())) {
						// a variable that appears in variables list but not in
						// invariant variables list is variant then we do no
						// need to do anything
					} else {
						// this is the first time we face this variable then it
						// is declared outside the block
						// so we check if it is assigned to an invariant
						// variable
						if (_invariantVariables.containsKey(rightName.getIdentifier())) {
							// variable declared outside the scope that is
							// assigned to an invariant variable is and
							// invariant variable, so we add it to invariant
							// list and variables list
							addInvariantVariableDependency(leftName.getIdentifier(), rightName
									.getIdentifier());
						} else {
							// since it is assigned to variant variable then add
							// the variable to variables list which will mark it
							// is variant
							_variables.add(leftName.getIdentifier());
						}
					}

				}
			}

			// check if left side is an array access
			if (leftExp instanceof ArrayAccess) {
				// cast array access
				ArrayAccess access = (ArrayAccess) leftExp;

				// check if array access is invariant
				if (isInvariantVariable(access)) {
					// left side is invariant - check if right side is
					if (isInvariantVariable(rightExp)) {
						// add left side variable to invariant list with right
						// side as dependent
						addInvariantVariableDependency(access.toString(), rightName.getIdentifier());
						// add left side array access with its index variables
						// to the invariant variables
						ArrayList<String> aaVar = new ArrayList<String>();
						getArrayAccessIndexVariablesList(aaVar, access);
						addInvariantVariableDependencyList(access.toString(), aaVar);
					} else {
						// if right side is variant then we remove right side
						// from invariant list
						removeInvariantVariable(access.toString());
					}

				} else {
					// left side is variant then remove it if it is in invariant
					// list
					removeInvariantVariable(access.toString());
				}
			}
		}

		// check if this is an infix expression
		if (rightExp instanceof InfixExpression) {
			// check when left side of the assignment is a simple name
			if (node.getLeftHandSide() instanceof SimpleName) {
				String leftVar = ((SimpleName) node.getLeftHandSide()).getIdentifier();

				InfixExpression iExpression = (InfixExpression) rightExp;
				// check if left operand and right operand is invariant or
				// constant
				boolean result = false;
				// saves infix expression variables list
				ArrayList<String> infVariables = new ArrayList<String>();

				if (isInvariantVariable(iExpression.getLeftOperand())
						&& isInvariantVariable(iExpression.getRightOperand())) {
					// check if for all extended operations are invariant
					// expression
					result = true;
					// add the left and right variables to the list if needed
					addVariableNameToList(infVariables, iExpression.getLeftOperand());
					addVariableNameToList(infVariables, iExpression.getRightOperand());

					List list = iExpression.extendedOperands();
					for (Iterator iterator = list.iterator(); iterator.hasNext();) {
						Expression expr = (Expression) iterator.next();
						// if this is a variable then add it to variables list
						// so it can be added as a dependent

						addVariableNameToList(infVariables, expr);

						if (!isInvariantVariable(expr)) {
							result = false;
						}
					}
				}
				if (result) {
					// this is an invariant expression then we add the
					// variable to the invariant list
					// check if the variable it self appears in the variables
					// list, that is dependent on itself
					if (infVariables.contains(leftVar)) {
						// if the variable is dependent on itself then its
						// already in the invariant list but it is not invariant
						// so we delete it
						removeInvariantVariable(leftVar);

					} else {
						// if the variable is not dependent on itself then add
						// it to invariant dependent list
						addInvariantVariableDependencyList(leftVar, infVariables);
					}

				} else {
					// if the result is false that is the infix expression
					// is not invariant then remove the variable from the
					// invariant list
					removeInvariantVariable(leftVar);
				}
			}

			// check when left side of the assignment is an array access
			if (node.getLeftHandSide() instanceof ArrayAccess) {
				ArrayAccess access = (ArrayAccess) node.getLeftHandSide();
				// check if array access is invariant
				if (isInvariantVariable(access)) {
					// Left side is invariant - check right side
					InfixExpression iExpression = (InfixExpression) rightExp;
					// check if left operand and right operand is invariant or
					// constant
					boolean result = false;
					// saves infix expression variables list
					ArrayList<String> infVariables = new ArrayList<String>();

					if (isInvariantVariable(iExpression.getLeftOperand())
							&& isInvariantVariable(iExpression.getRightOperand())) {
						// check if for all extended operations are invariant
						// expression
						result = true;
						// add the left and right variables to the list if
						// needed
						addVariableNameToList(infVariables, iExpression.getLeftOperand());
						addVariableNameToList(infVariables, iExpression.getRightOperand());

						List list = iExpression.extendedOperands();
						for (Iterator iterator = list.iterator(); iterator.hasNext();) {
							Expression expr = (Expression) iterator.next();
							// if this is a variable then add it to variables
							// list
							// so it can be added as a dependent
							addVariableNameToList(infVariables, expr);

							if (!isInvariantVariable(expr)) {
								result = false;
							}
						}
					}
					if (result) {
						// this is an invariant expression then we add the
						// variable to the invariant list
						// check if the variable it self appears in the
						// variables
						// list, that is dependent on itself
						if (infVariables.contains(access.toString())) {
							// if the variable is dependent on itself then its
							// already in the invariant list but it is not
							// invariant
							// so we delete it
							removeInvariantVariable(access.toString());

						} else {
							// if the variable is not dependent on itself then
							// add
							// it to invariant dependent list
							addInvariantVariableDependencyList(access.toString(), infVariables);
							// add left side array access with its index
							// variables
							// to the invariant variables
							ArrayList<String> aaVar = new ArrayList<String>();
							getArrayAccessIndexVariablesList(aaVar, access);
							addInvariantVariableDependencyList(access.toString(), aaVar);

						}

					} else {
						// if the result is false that is the infix expression
						// is not invariant then remove the variable from the
						// invariant list
						removeInvariantVariable(access.toString());
					}

				} else {
					// array access is not an invariant variable then we remove
					// it from invariant list if found
					removeInvariantVariable(access.toString());
					// but we check the right side of the assignment for
					// invariant variables and add them to the list of invariant
					// variables if found
					addInfixExpressionInvariants((InfixExpression) rightExp);
				}

			}
		}

		// check if this is an array access
		if (rightExp instanceof ArrayAccess) {
			ArrayAccess access = (ArrayAccess) rightExp;
			// left hand side is a simple variables
			if (node.getLeftHandSide() instanceof SimpleName) {
				// get left variable name
				String leftVar = ((SimpleName) node.getLeftHandSide()).getIdentifier();

				// check if right side is invariant
				if (isInvariantVariable(access)) {
					// check first if left side is variant
					if (_variables.contains(leftVar) && !_invariantVariables.containsKey(leftVar)) {
						// variable is already variant so we do not need to do
						// anything
					} else {
						// add the variable with its dependent list to the
						// invariant map
						addInvariantVariableDependency(leftVar, access.toString());

						// get dependent list
						ArrayList<String> aaVar = new ArrayList<String>();
						getArrayAccessIndexVariablesList(aaVar, access);
						// add array access to invariant list
						addInvariantVariableDependencyList(access.toString(), aaVar);

					}

				} else {
					// right side is not invariant
					// remove left side from invariant map if found
					removeInvariantVariable(leftVar);
				}
			}
			// check when left side is an array access too
			if (node.getLeftHandSide() instanceof ArrayAccess) {
				// get left side array access
				ArrayAccess leftAccess = (ArrayAccess) node.getLeftHandSide();
				// check if left side is invariant
				if (isInvariantVariable(leftAccess)) {
					// check if right side is invariant
					if (isInvariantVariable(access)) {
						// both right and left side are invariant so
						// add left side to invariant list with right side as
						// dependent
						addInvariantVariableDependency(leftAccess.toString(), access.toString());
						// add right side and its indexes as invariants
						ArrayList<String> aaVar = new ArrayList<String>();
						getArrayAccessIndexVariablesList(aaVar, access);
						addInvariantVariableDependencyList(access.toString(), aaVar);
						// add left side and its indexes as invariants
						ArrayList<String> aaVar1 = new ArrayList<String>();
						getArrayAccessIndexVariablesList(aaVar1, leftAccess);
						addInvariantVariableDependencyList(leftAccess.toString(), aaVar1);

					} else {
						// right side is not invariant but left side is
						// invariant which makes left side variant
						// so we remove left side from invariant list
						removeInvariantVariable(leftAccess.toString());
					}

				} else {
					// if left side is invariant then we remove it from list if
					// it is there
					removeInvariantVariable(leftAccess.toString());
				}
			}
		}

		return super.visit(node);
	}

	// ===================== Help methods
	/**
	 * Adds a variable to the invariant list with the supplied dependent if
	 * applicable.
	 * 
	 * @param key
	 *            - variable name
	 * @param dependent
	 *            - dependent variable name
	 */
	private void addInvariantVariableDependency(String key, String dependent) {
		// try to get invariant variables of key
		ArrayList<String> tmp = _invariantVariables.get(key);
		// if variable is not invariant and does not appear in variables list
		// then add it
		if (!_variables.contains(key)) {
			_variables.add(key);
		}

		// check if variable is invariant i.e. was found in invariant list
		if (tmp != null) {
			// add dependent to variable dependency list
			if (dependent != null && !tmp.contains(dependent)) {
				tmp.add(dependent);
			}
		} else {
			// if variable was not found in invariant list then add it
			tmp = new ArrayList<String>();
			if (dependent != null) {
				tmp.add(dependent);
			}
			// add variable to map
			_invariantVariables.put(key, tmp);
		}
	}

	/**
	 * Handles addition of multiple dependents by adding them one by one.
	 *
	 * @param key - the key variable
	 * @param dependent list of dependent variables
	 */
	private void addInvariantVariableDependencyList(String key, ArrayList<String> dependent) {
		if (dependent.size() > 0) {
			for (Iterator<String> iterator = dependent.iterator(); iterator.hasNext();) {
				String var = (String) iterator.next();
				// add dependent and key
				addInvariantVariableDependency(key, var);
			}
		} else {
			addInvariantVariableDependency(key, null);
		}

	}

	/**
	 * This method iterate over the infix expression elements and add them to
	 * the invariant list if they are invariant.
	 *
	 * @param infixExpression the infix expression
	 */
	@SuppressWarnings("unchecked")
	private void addInfixExpressionInvariants(InfixExpression infixExpression) {
		// add left side of infix expression
		if (infixExpression.getLeftOperand() instanceof SimpleName) {
			SimpleName name = (SimpleName) infixExpression.getLeftOperand();
			// check if already appears in invariant list but not in invariant list
			if( _variables.contains(name.getIdentifier()) && ! _invariantVariables.containsKey(name.getIdentifier()))
			{
				
			}else{
				addInvariantVariableDependency(name.getIdentifier(), null);
			}
		}

		// add left side of infix expression
		if (infixExpression.getLeftOperand() instanceof ArrayAccess) {
			ArrayAccess name = (ArrayAccess) infixExpression.getLeftOperand();
			
			// check if already appears in invariant list but not in invariant list
			if( _variables.contains(name.toString()) && ! _invariantVariables.containsKey(name.toString()))
			{
				
			}else{
				addInvariantVariableDependency(name.toString(), null);
			}
		}

		// add right side of infix expression
		if (infixExpression.getRightOperand() instanceof SimpleName) {
			SimpleName name = (SimpleName) infixExpression.getRightOperand();
			// check if already appears in invariant list but not in invariant list
			if( _variables.contains(name.getIdentifier()) && ! _invariantVariables.containsKey(name.getIdentifier()))
			{
				
			}else{
				addInvariantVariableDependency(name.getIdentifier(), null);
			}
		}

		// add left right of infix expression
		if (infixExpression.getRightOperand() instanceof ArrayAccess) {
			ArrayAccess name = (ArrayAccess) infixExpression.getRightOperand();
			// check if already appears in invariant list but not in invariant list
			if( _variables.contains(name.toString()) && ! _invariantVariables.containsKey(name.toString()))
			{
				
			}else{
				addInvariantVariableDependency(name.toString(), null);
			}
		}

		// loop over all extended infix expression
		List extended = infixExpression.extendedOperands();
		for (Iterator iterator = extended.iterator(); iterator.hasNext();) {
			Expression exp = (Expression) iterator.next();

			// check for simple name
			if (exp instanceof SimpleName) {
				SimpleName name = (SimpleName) exp;
				// check if already appears in invariant list but not in invariant list
				if( _variables.contains(name.getIdentifier()) && ! _invariantVariables.containsKey(name.getIdentifier()))
				{
					
				}else{
					addInvariantVariableDependency(name.getIdentifier(), null);
				}
			}
			// check for array access
			if (exp instanceof ArrayAccess) {
				ArrayAccess name = (ArrayAccess) exp;
				// check if already appears in invariant list but not in invariant list
				if( _variables.contains(name.toString()) && ! _invariantVariables.containsKey(name.toString()))
				{
					
				}else{
					addInvariantVariableDependency(name.toString(), null);
				}
			}
		}
	}

	/**
	 * This method removes a variable from the invariant variables map it also
	 * removes any dependent variables from the invariant list.
	 *
	 * @param key - variable to be removed
	 */
	private void removeInvariantVariable(String key) {
		ArrayList<String> value;
		// remove from invariant map
		value = _invariantVariables.remove(key);

		// check if value is not found in invariants map
		if (value == null) {
			// add it to variables map in order to assure its appearance as a
			// variant
			if (!_variables.contains(key)) {
				_variables.add(key);
			}
		}

		// remove all dependent variables
		Set<String> variables = new HashSet<String>();
		variables.addAll(_invariantVariables.keySet());
		// go through all variables
		ArrayList<String> dependents;
		for (Iterator<String> iterator = variables.iterator(); iterator.hasNext();) {
			String var = (String) iterator.next();
			// check if key is a dependent of the value
			dependents = _invariantVariables.get(var);
			if (dependents != null && dependents.contains(key)) {
				// if variable is dependent on our removed variable then remove
				// it from invariant list
				removeInvariantVariable(var);
			}
		}
	}

	/**
	 * Check if the right side of the assignment is a constant value that is
	 * some literal type.
	 *
	 * @param exp Expression
	 * @return TRUE - A constant Value FALSE - Variable or expression
	 */
	private boolean isConstantValue(Expression exp) {
		// check for null
		if (exp == null) {
			return false;
		}
		// check for different types of literals
		if (exp instanceof NumberLiteral || exp instanceof StringLiteral
				|| exp instanceof BooleanLiteral || exp instanceof NullLiteral
				|| exp instanceof CharacterLiteral) {
			// if it is a literal type
			return true;
		}

		return false;
	}

	/**
	 * Check if the variable is invariant at the current line of the code, that
	 * is, it check according to the following conditions 1) if is in invariant
	 * variables code - return true 2) if is not in invariant variables map but
	 * is inside variables array list - return false 3) if is not in both
	 * invariant or variables then this is a new variable and is an invariant
	 * variable so we add it to the invariant list.
	 *
	 * @param exp the exp
	 * @return TRUE - invariant, FALSE - variant
	 */
	private boolean isInvariantVariable(Expression exp) {
		// check if expression is constant
		if (isConstantValue(exp)) {
			return true; // constant value is always invariant
		}

		// check if it is a variable
		if (exp instanceof SimpleName) {

			String var = ((SimpleName) exp).getIdentifier();
			// check if variable appears in invariant map
			if (_invariantVariables.containsKey(var)) {
				return true;
			} else {
				// check if variables appear in variables list i.e. is variant
				if (_variables.contains(var)) {
					return false;
				} else {
					// this means that this is the first time that we reference
					// this
					// variable inside the block so we add it to the invariant
					// list
					_variables.add(var);
					addInvariantVariableDependency(var, null);
					return true;
				}
			}
		}

		// check for array access
		if (exp instanceof ArrayAccess) {
			ArrayAccess arAccess = (ArrayAccess) exp;
			// check if both array and and index are invariant
			if (isInvariantVariable(arAccess.getArray())
					&& isInvariantVariable(arAccess.getIndex())) {
				return true;
			} else {
				return false;
			}
		}
		return false;
	}

	/**
	 * Find all variable array access indexes.
	 *
	 * @param aaVar array list for storing variable array access indexes
	 * @param access array access node
	 * @return the array access index variables list
	 */
	private void getArrayAccessIndexVariablesList(ArrayList<String> aaVar, ArrayAccess access) {
		// check if the first level array access is a variable
		if (access.getIndex() instanceof SimpleName) {
			SimpleName name = (SimpleName) access.getIndex();
			aaVar.add(name.getIdentifier());
		}

		// check if this is a nested array access call
		if (access.getArray() instanceof ArrayAccess) {
			getArrayAccessIndexVariablesList(aaVar, (ArrayAccess) access.getArray());
		}
	}

	/**
	 * Returns the array name by taking the array access.
	 *
	 * @param access the access
	 * @return String the array name
	 */
	private String getArrayName(ArrayAccess access) {
		if (access.getArray() instanceof SimpleName) {
			return ((SimpleName) access.getArray()).getIdentifier();
		} else {
			if (access.getArray() instanceof ArrayAccess) {
				return getArrayName((ArrayAccess) access.getArray());
			}
		}
		return null;
	}

	/**
	 * Adds the variable name to the list according to its type, Simple Name or
	 * Array Access.
	 *
	 * @param infVariables List for holding the variable names
	 * @param exp the required expression
	 */
	private void addVariableNameToList(ArrayList<String> infVariables, Expression exp) {
		// check if simple name
		if (exp instanceof SimpleName) {
			infVariables.add(((SimpleName) exp).getIdentifier());
		}
		// check if this is an array access variable
		if (exp instanceof ArrayAccess) {
			infVariables.add(((ArrayAccess) exp).toString());
		}

	}
}
